/* ***********************************************************/
/* Copyright (c)2007-2017 by Progress Software Corporation        */
/*                                                           */
/* All rights reserved.  No part of this program or document */
/* may be  reproduced in  any form  or by  any means without */
/* permission in writing from Progress Software Corporation. */
/*************************************************************/

/* this file handles compiles for OpenEdge version 10.1B and
*  higher using the new language features for a compile error
*  messages.
*/


/* compile.p - Eclipse compile procedure */
define input  parameter cPrm            as character no-undo.
define output parameter compileResult    as longchar no-undo.

define new global shared variable OEIDE_Context    as handle no-undo.
function getProjectWorkDirectory returns character 	(  ) in OEIDE_Context.
function getProjectName returns character   (  ) in OEIDE_Context.

define new shared variable XCodeKey         as character no-undo initial "".
define new shared variable XRefFile         as character no-undo initial "".
define new shared variable CodeListFile     as character no-undo initial "".
define new shared variable DebugListFile    as character no-undo initial "".
define new shared variable RCodePath        as character no-undo initial "".
define new shared variable SourceFileFull   as character no-undo initial "".

define new shared stream WEBSTREAM.  /** Allow WebSpeed compile */
define stream CMPLSTREAM.
define stream s1.


define variable c1               as character no-undo.

/** Compile variables **/
define variable cNewfile         as character no-undo initial "".
define variable SourceFile       as character no-undo initial "".
define variable cSpeedfile       as character no-undo initial "".
define variable LiteralOptions   as character no-undo initial "".
define variable strictOptions    as character no-undo initial "".
define variable CheckSyntax      as logical   no-undo initial false.
define variable SaveRCode        as logical   no-undo initial false.
define variable SaveXRef         as logical   no-undo initial false.
define variable CodeListing      as logical   no-undo initial false.
define variable DebugListing     as logical   no-undo initial false.
define variable SyntaxFile       as character no-undo initial "". 
define variable CompileTempFile  as character no-undo initial "". 

define variable XRefXmlDirectory as character no-undo initial "". 

define variable lHTMLMapped      as logical   no-undo initial false.
/* to verify existance of HTMlMapped file for the html/speedscript file */
define variable htmlMappedFile    as character no-undo initial "".
define variable PreprocessFlag   as logical   no-undo initial false.
define variable PreprocessFile   as character no-undo initial "".
/* Special flag to indicate the current speed sccript file is include file.*/
define variable isInclude  as logical   no-undo initial false.
/* Special flag to indicate the current resource is speed script file.*/
define variable isSpeedScriptFile as logical no-undo.
define variable isHTMLResource as logical NO-UNDO.
define variable idx as integer no-undo.

&GLOBAL-DEFINE COMMAND_SEPARATOR CHR(3)
	&GLOBAL-DEFINE OPTION_SEPARATOR CHR(4)


run separateOptions.

if checksyntax then 
do:
    assign
        SaveRCode      = false
        SourceFileFull = SyntaxFile.
end.
else 
do:
    SourceFileFull = SourceFile.
end.
CompileTempFile = getProjectWorkDirectory() + "/" + getProjectName() + "/../compile_tmp_" + STRING(ETIME) + ".p".

IF SourceFile MATCHES "*~~.html" OR SourceFile MATCHES "*~~.htm" 
OR SourceFile MATCHES "*~~.xhtml" OR SourceFile MATCHES "*~~.htpl" OR SourceFile MATCHES "*~~.wml"
OR SourceFile MATCHES "*~~.shtml" OR SourceFile MATCHES "*~~.shtm" THEN DO:
    
isHTMLResource = true.

idx = r-index(SourceFile, ".").    
assign 
    htmlMappedFile = substring(SourceFile, 1, idx, "CHARACTER") + "w".
/* Verify availability of html mapped file */
FILE-INFORMATION:file-name = htmlMappedFile.
if FILE-INFORMATION:full-pathname > "" then
    lHTMLMapped = true.
else
    lHTMLMapped = false.
    
/* In case of check syntax operation for HTML file, set cNewFile to temporary location.*/    
if checksyntax or NOT lHTMLMapped then 
	do:
    	idx = r-index(SourceFileFull, ".").    
    	assign 
        	cNewFile = substring(SourceFileFull, 1, idx, "CHARACTER") + "w".
end.

/* Don't compile HTML mapped html files */
if not lHTMLMapped then
do:
    run webutil/e4gl-gen.r (input SourceFileFull, 
        input-output cSpeedFile,
        input-output cNewfile).
    if index(cSpeedFile, "include") > 0   then
    do: 
        assign 
            isInclude = true.
    end.
end.
else if checksyntax then
    do:
        os-copy VALUE(FILE-INFORMATION:full-pathname) VALUE(cNewfile).
    end.       
                                  
assign 
    SourceFileFull = cNewfile.
end.

/* Don't process file if it is speedscript file with "include" meta tag. */
IF isInclude AND isSpeedScriptFile AND NOT CheckSyntax THEN
DO:
    return.    
END.


run CreateCompileTemp.


run VALUE(CompileTempFile) "".


os-delete VALUE(CompileTempFile).

run parseErrors( output compileResult).

/* cleanup */
if checksyntax or (isHTMLResource and not lHTMLMapped) then 
do:
    os-delete VALUE(SourceFileFull).
end.


/*
  this section initializes all the variables needed by the compile statement generator
  it is the responsability of this section to make sure all paths use forward-slashes ("/")

*/
procedure separateOptions.

    define variable i           as integer   no-undo.
    
    define variable OptionCount as integer   no-undo.
    define variable cOption     as character no-undo.
    define variable optName     as character no-undo.
    define variable optValue    as character no-undo.
    
    OptionCount = num-entries(cPrm, {&COMMAND_SEPARATOR}).
    /*
        Should we reset to Unknown. If so we need to use. Whose responsibility to 
        reset this?
    */
    security-policy:xcode-session-key  =  ?.    
    do i = 1 to OptionCount:
        cOption = entry(i, cPrm, {&COMMAND_SEPARATOR}).
        OptName = entry(1, cOption, {&OPTION_SEPARATOR}).
        optValue = entry(2, cOption, {&OPTION_SEPARATOR}).
        case optName:
            when "LITERAL" then
                LiteralOptions = OptValue.
            when "OPTIONS" then
                strictOptions = OptValue.
            when "FILE" then
                SourceFile = OptValue.
            when "XREF" then
                if OptValue = "TRUE" then
                    SaveXRef = true.
            when "XCODEKEY" then
                XCodeKey = optValue.
            when "XCODESESSIONKEY" then 
            run VALUE(replace(optValue, "~\", "/")).
            when "SAVEINTO" then 
                do:
                    if OptValue <> "" then 
                    do:
                        RCodePath = optValue.
                    end.
                end.
            when "SAVE" then 
                do:
                    if optValue = "true" then
                        SaveRCode = true.           
                end.
            when "CHECKSYNTAX" then 
                do:
                    checksyntax = true.
                end.
            when "TEMPFILE" then 
                do:
                    SyntaxFile = optValue.
                    SyntaxFile = replace(SyntaxFile, "~\", "/").                
                end.
            when "DEBUGLIST" then 
                do:
                    DebugListing = true.
                    DebugListFile = optValue.
                end.
            when "CODELISTING" then 
                do:
                    CodeListing = true.
                    CodeListFile = optValue.
                end.
            when "XREFFILE" then 
                do:
                    XRefFile = optValue.
                end.
            when "CODELISTINGFILE" then 
                do:
                    CodeListFile = optValue.
                end.
            when "DEBUGLIST" then 
                do:
                    DebugListFile = optValue.
                end.
            when "XREFXML" then 
                do:
                    XRefXmlDirectory = optValue.
                end.
            when "PREPROCESS" then 
                do:
                    PreprocessFlag = true.
                    PreprocessFile = optValue.
                end.
            when "SPEEDSCRIPTFILE" then
                do:
                    isSpeedScriptFile = true.
                end.
        end case.
    
    end.
    
end procedure.

/*
    This builds the file that gets run to compile an editor file
    This is done to trap any types of errors as well as be able
    to specify all the possible compile syntax
*/  
    
procedure CreateCompileTemp.
    output STREAM CMPLSTREAM TO VALUE(CompileTempFile).

    put stream CMPLSTREAM unformatted "DEFINE SHARED VARIABLE XCodeKey         AS CHARACTER NO-UNDO." skip.
    put stream CMPLSTREAM unformatted "DEFINE SHARED VARIABLE SourceFileFull   AS CHARACTER NO-UNDO." skip.
    put stream CMPLSTREAM unformatted "DEFINE SHARED VARIABLE CodeListFile     AS CHARACTER NO-UNDO." skip.
    put stream CMPLSTREAM unformatted "DEFINE SHARED VARIABLE DebugListFile    AS CHARACTER NO-UNDO." skip.
    put stream CMPLSTREAM unformatted "DEFINE SHARED VARIABLE RCodePath        AS CHARACTER NO-UNDO." skip.

    put stream CMPLSTREAM unformatted skip.
    put stream CMPLSTREAM unformatted "COMPILE VALUE(SourceFileFull) ".
    if LiteralOptions <> "" then
        put stream CMPLSTREAM unformatted LiteralOptions + " ".
    if strictOptions <> "" then
        put stream CMPLSTREAM unformatted 'OPTIONS "' strictOptions '" '.
    if SaveRCode then 
    do:
        put stream CMPLSTREAM unformatted "SAVE ".
        if RCodePath <> "" then
            put stream CMPLSTREAM unformatted "INTO VALUE(RCodePath) ".
    end.
    if XCodeKey <> "" then
        put stream CMPLSTREAM unformatted "XCODE XCodeKey ".    
    if SaveXRef then
        put stream CMPLSTREAM unformatted "XREF VALUE(XRefFile) ".
    if CodeListing then
        put stream CMPLSTREAM unformatted "LISTING VALUE(CodeListFile) ".       
    if DebugListing then
        put stream CMPLSTREAM unformatted "DEBUG-LIST VALUE(DebugListFile) ".
    if XrefXmlDirectory <> "" then
        put stream CMPLSTREAM unformatted 'XREF-XML "' XrefXmlDirectory '" '.
    if PreprocessFlag then
        put stream CMPLSTREAM unformatted 'PREPROCESS "' PreprocessFile '" '.

    put stream CMPLSTREAM unformatted "NO-ERROR." skip.
    output STREAM CMPLSTREAM CLOSE.

end procedure.

&scoped-define COMPILE_OK  0
&scoped-define COMPILE_WARNING 1
&scoped-define COMPILE_ERROR 2
&scoped-define COMPILE_FAILURE 4
&scoped-define COMPILE_STOPPED 8
&scoped-define COMPILE_MESSAGES 16

/*
 This reads the error messages from the compiler handle if there are any
and then puts them along with the line number and file name where the error
occured into a format that can be read by Developer Studio.  This only works with
10.1B and forward.
*/
procedure parseErrors.

    define output parameter compileResult as longchar no-undo.

    define variable i                as integer   no-undo.
    define variable messageCount     as integer   no-undo.
    define variable msg              as character no-undo.
    define variable msgNumber        as integer   no-undo.
    define variable msgRow           as integer   no-undo.
    define variable fileName         as character no-undo.
    define variable msgType          as integer   no-undo.
    define variable msgColumn        as integer   no-undo.
    
    define variable compilerState    as integer no-undo.
    define variable compilerMessages as longchar no-undo.
    
    if compiler:warning then
        compilerState = compilerState + {&COMPILE_WARNING}.
    
    if compiler:error then
        compilerState = compilerState + {&COMPILE_ERROR}.
        
    if error-status:error then
        compilerState = compilerState + {&COMPILE_FAILURE}.
        
    if compiler:stopped then
        compilerState = compilerState + {&COMPILE_STOPPED}.
    
    if compiler:num-messages > 0 then 
    do:
        assign 
            messageCount  = compiler:num-messages
            compilerState = compilerState + {&COMPILE_MESSAGES}.
        
        do i = 1 to messageCount:
            assign 
                msg       = compiler:get-message(i)
                msgNumber = compiler:get-number(i)
                msgRow    = compiler:get-row(i)
                msgColumn = compiler:get-column(i)
                fileName  = compiler:get-file-name(i)
                msgType   = compiler:get-message-type(i).
            if msgRow = ? then
                msgRow = 0.
            if msgNumber = ? then
                msgNumber = 0.
            if fileName = ? then
                fileName = SourceFile.
            if msgColumn = ? then
                msgColumn = 0.
            compilerMessages = compilerMessages + fileName + {&OPTION_SEPARATOR} + string(msgRow) + {&OPTION_SEPARATOR} + string(msgColumn) + {&OPTION_SEPARATOR}+ msg + {&OPTION_SEPARATOR} + string(msgNumber) + {&OPTION_SEPARATOR} + string(msgType) + {&COMMAND_SEPARATOR}.
        end.
    end.
    else 
    do:
        /* this handles errors with the execution of the compile statement itself as opposed to problems with the code */
        /* things like a missing "save into" directory would fall into this category */
        assign
            messageCount  = error-status:num-messages.
        do i = 1 to messageCount:
            msgRow = compiler:error-row.
            msgColumn = compiler:error-column.
            if msgRow = ? then
                msgRow = 0.
            if msgColumn = ? then
                msgColumn = 0.
            msgNumber = error-status:get-number(i).
            msg = error-status:get-message(i).
            msgType = -1. /* -1 indicates unknown type, the java side will determine the severity based on if the compile failed or succeed */
            compilerMessages = compilerMessages + SourceFile + {&OPTION_SEPARATOR} + string(msgRow) + {&OPTION_SEPARATOR} + string(msgColumn) + {&OPTION_SEPARATOR} + msg + {&OPTION_SEPARATOR} + string(msgNumber) + {&OPTION_SEPARATOR} + string(msgType) + {&COMMAND_SEPARATOR}.
        end.
    end.

    compileResult = string(compilerState) + {&COMMAND_SEPARATOR} + compiler:class-type + {&COMMAND_SEPARATOR} + string(messageCount) + {&COMMAND_SEPARATOR} + compilerMessages.
end procedure.

