/* ***********************************************************/
/* Copyright (c) 2012-2016 by Progress Software Corporation       */
/*                                                           */
/* All rights reserved.  No part of this program or document */
/* may be  reproduced in  any form  or by  any means without */
/* permission in writing from Progress Software Corporation. */
/*************************************************************/
 
define input parameter phServer as handle no-undo.
     
define variable fSocketHandle    as handle    no-undo.
define variable aOk              as logical   no-undo.
define variable mResponse        as character no-undo.
define variable iSequence        as integer      no-undo.
define variable fLoggerHandle    as handle    no-undo. 
/* this temp-table only contains a single record and its buffer is used
   like an object to hold onto the options read from the command message.
*/   
define temp-table ttCommand no-undo
    field Name as character
    field Parameters as character
    field Scope as character
    field RequestID as int64
    field ResultIsLongChar as logical. 
 
/* declaration for protocol */
&SCOPED-DEFINE MSG_HEADER_SIZE 11
&SCOPED-DEFINE PACKET_HEADER_SIZE 5
&SCOPED-DEFINE MSG_HEADER_BEGIN 254
&SCOPED-DEFINE MSG_HEADER_END   253
&SCOPED-DEFINE PACKET_BEGIN     252
&SCOPED-DEFINE PACKET_END       251
&SCOPED-DEFINE MSG_END          250

/* declarations for command identifiers */

/* the name of the program to run */
&SCOPED-DEFINE COMMAND_PROGRAM 1

/* the parameters to pass to the program that is being run */ 
&SCOPED-DEFINE COMMAND_PARAMETERS 2

/* the scope of the command.  Must be already registered */
&SCOPED-DEFINE COMMAND_SCOPE 3

/* use alternate run statement to allow longchar return value */
&SCOPED-DEFINE COMMAND_USES_LONGCHAR 4

function log returns character 
    ( input msg as character) in phServer.

function getProjectName returns character 
    ( ) in phServer.

function getServerDirectory returns character 
    ( ) in phServer.
 
&global-define DEFAULT-SERVER-PORT 3333
&global-define MAX-CONNECTION-FAILURES 3

&IF '{&WINDOW-SYSTEM}' = 'GUI':U &THEN
  {src/adm2/globals.i}
&ENDIF

/******* Initialization ****/
 
create socket fSocketHandle.

run ConnectToServer no-error.

if not aOK or error-status:error or not fSocketHandle:connected() then 
do:
    log( "Error Status: " + string(error-status:error)).
    log( "Return Value: " + return-value ).
    log( "Error Message: " + error-status:get-message(1)).
    log( "aOk variable: " + string(aOk)).
    log( "Socket connected: " + string(fSocketHandle:connected())).
    return error.
end.

/***************************** functions ******************************/
function log returns character 
    ( input msg as character).
    
    if valid-handle(fLoggerHandle) then 
    do:
    
        dynamic-function("log" in fLoggerHandle, msg).
    end.
    
end function.

/* **********************  Internal Procedures  *********************** */
/* hide as ade object in the persistent procedure viewer  */
procedure adepersistent:
end.

procedure Disconnect.
    if fSocketHandle:connected() then 
    do:
        fSocketHandle:disconnect().
    end. 
end procedure.

procedure ConnectToServer.
    define variable cPort as character no-undo.

    cPort = os-getenv("OEA_PORT").

    if cPort = "" or cPort = ? then cPort = "{&DEFAULT-SERVER-PORT}".  /* Default port */
    
    log( "Connecting to eclipse project").
    aOk = fSocketHandle:connect("-S " + cPort) no-error.
    log( error-status:get-message(1)).
     
    if not aOK then
        return error "Connection to eclipse project failed on port " + cPort.
    else 
    do:
        run SendConnectionGreeting no-error.
        if error-status:error = true then 
        do:
            return error return-value.
        end.
    end.

end procedure.

procedure SendConnectionGreeting:
    define variable greeting    as character no-undo.
    define variable projectname as character no-undo.  
    define variable msg         as character no-undo initial "Project name not passed in environment".
  
    fSocketHandle:set-read-response-procedure("ReceiveCommand", this-procedure).   
    projectname = getProjectName().
    if projectname = "unknown" then 
    do:
        log(msg).
        aOk = false.
        return error msg.
    end.
    greeting = "CLIENT:" + projectname.
    log( "Connected").
  
    run WriteToSocket(0, greeting) no-error.
    
    if error-status:error = true then 
    do:
        return error return-value.
    end.
end.

procedure SendRequest:
    define input parameter requestid as integer  no-undo.
    define input parameter packet    as longchar no-undo.    
    if length(packet,"RAW") < 28000 then 
            log( "SendRequest " + string(requestid) + " : " + string(packet)).
    else if packet = ? then 
            log( "SendRequest " + string(requestid) + " : " +  "<value is unknown>").
    else   
            log( "SendRequest " + string(requestid) + " : " + "<value larger than 28K - " + string(length(packet,"RAW"),"RAW") + " >").
    
    run WriteToSocketInternal(requestid,0,packet,no).
end procedure.

procedure SendWaitRequest:
    define input parameter requestid as integer  no-undo.
    define input parameter packet    as longchar no-undo.    
    define variable hdl as handle no-undo.
    define output parameter pcResult as character no-undo.    
    
    run value(getServerDirectory()+ "_wait.p") persistent set hdl.
    
    if length(packet,"RAW") < 28000 then 
            log( "SendWaitRequest " + string(requestid) + " : " + string(packet)).
    else if packet = ? then 
            log( "SendWaitRequest " + string(requestid) + " : " +  "<value is unknown>").
    else   
            log( "SendWaitRequest " + string(requestid) + " : " + "<value larger than 28K - " + string(length(packet,"RAW"),"RAW") + " >").
  
    run WriteToSocketInternal(requestid,integer(hdl),packet,no).
    
    /* wait on HandleReply */ 
    run wait in hdl.
       
    /* mResponse is set in handleReply */ 
    pcResult = mResponse.
end procedure.

procedure WriteToSocket:
    define input parameter requestid as integer no-undo.
    define input parameter packet    as longchar no-undo.
    run WriteToSocketInternal(requestid,0,packet,yes).
end procedure.

procedure setLoggerHandle:
    define input parameter hLogger as handle no-undo.
    fLoggerHandle = hLogger.
end.   

/** handles writing of a message to the eclipse session
 *
 * Message format:
   - MSG_HEADER_BEGIN    - 1 byte      - CHR(254)
   - request/reply       - 1 byte      - 0/1
   - Server requestId    - 4 bytes
   - Client requestId    - 4 bytes
   - MSG_HEADER_END      - 1 byte      - CHR(253)
   - PACKET_BEGIN        - 1 byte      - CHR(252)
   - packetLength        - 4 bytes
   - data                - priceless
   - PACKET_END          - 1 byte      - CHR(251)
   - MSG_END             - 1 byte      - CHR(250)
*/
procedure WriteToSocketInternal private:
    define input parameter requestid as integer no-undo.
    define input parameter replyid   as integer no-undo.
    define input parameter packet    as longchar no-undo.
    define input parameter response  as logical no-undo.
   
    define variable messageHeader  as memptr    no-undo.
    define variable packetBuffer   as memptr    no-undo.
    define variable messageTrailer as memptr    no-undo.
  
    define variable packetLength   as integer   no-undo.
    define variable ok             as logical   no-undo.
    define variable msg            as character no-undo.
     
    if packet = ? then
        packet = "?".
    
    ok = valid-handle(fSocketHandle).
    
    if ok = true  then 
    do:
        ok = fSocketHandle:connected().
        
        if ok = true then 
        do:
            SET-SIZE(messageHeader)     = {&MSG_HEADER_SIZE}.
            PUT-BYTE(messageHeader, 1)  = {&MSG_HEADER_BEGIN}.
            PUT-BYTE(messageHeader, 2)  = if response then 1 else 0.            
            PUT-LONG(messageHeader, 3)  = RequestId.
            /*
            do on error undo, throw:
                iSequence = iSequence + 1.
                catch e as Progress.Lang.Error :
                	iSequence = 1.	
                end catch.
            end.
            */
            PUT-LONG(messageHeader, 7)  = replyid. 
            PUT-BYTE(messageHeader, 11) = {&MSG_HEADER_END}.
      
            ok = fSocketHandle:write (messageHeader,1,{&MSG_HEADER_SIZE}) no-error.      
            if ok = true then 
            do:
                /* Bug - OE00198914 */
                /* Modified By - grkumar
                   Here a check is put if the SESSION:CPINTERNAL is UTF-8 or not. If 
                   not then the message is converted in UTF-8 encoding format.
                   Another change is in the calculation of packetLength in the else
                   condition below. One more attribute is given i.e., "RAW" in the LENGTH()
                   function which returns the required length in Bytes. If we don't specify
                   any attribute then by default the length is returned in Characters which
                   may result in loss of some characters at the end of the stream resulting 
                   in error on the Java side.
                   HD - changed to always use utf-8 cpinternal is irrelevant
                   OE00221236 Issues displaying UTF-8 data in IE and chrom
                   
                */    
                
                define variable mpacket as memptr no-undo.
                set-byte-order(mpacket) = 3.
                COPY-LOB packet TO mpacket CONVERT TARGET CODEPAGE "utf-8".
                packetLength = GET-SIZE(mpacket).
                SET-SIZE(packetBuffer) = packetLength + {&PACKET_HEADER_SIZE} + 1.
                PUT-BYTE(packetBuffer, 1) = {&PACKET_BEGIN}.
                PUT-LONG(packetBuffer, 2) = packetLength.
                PUT-BYTES(packetBuffer, 6) = mpacket.

                PUT-BYTE(packetBuffer, {&PACKET_HEADER_SIZE} + packetLength + 1) = {&PACKET_END}.
                
                ok = fSocketHandle:write (packetBuffer,1, {&PACKET_HEADER_SIZE} + packetLength + 1) no-error.
          
                if ok = true then 
                do:
                    SET-SIZE(messageTrailer) = 1.   
                    PUT-BYTE(messageTrailer, 1) = {&MSG_END}.
                    ok = fSocketHandle:write (messageTrailer,1, 1) no-error.
                end.
    
            end.
      
        end.
    
    end.

    SET-SIZE(messageHeader)  = 0.
    SET-SIZE(packetBuffer)   = 0.    
    SET-SIZE(messageTrailer) = 0.
    
    if OK <> true then 
    do:
        msg = error-status:get-message(1).
        run QUIT in phserver(msg).
        return error msg.
    end.
    
    return "".
    
end.

procedure extractCommand:
    define input parameter requestData  as memptr    no-undo.
    define parameter buffer Command for ttCommand.
    
    define variable tlvCount as integer no-undo.
    define variable tlvType as integer no-undo.
    define variable tlvSize as integer no-undo.
    define variable tlvValue as character no-undo.
     
    define variable iPos    as integer no-undo initial 1.
    define variable tlvEntry as integer no-undo.

    run ClearReturnValue no-error.

    do on error undo, leave
        on stop undo, leave:
        iPos = 1.
     
        tlvCount = get-long(requestData, iPos).
        iPos = iPos + 4.
        
        do tlvEntry = 1 to tlvCount:
            tlvType = get-long(requestData, iPos).
            iPos = iPos + 4.
            tlvSize = get-long(requestData, iPos).
            iPos = iPos + 4.
            if (tlvSize > 0) then 
            do:
                tlvValue = get-string(requestData, iPos, tlvSize).
                iPos = iPos + tlvSize.
            end.
            else do:
                tlvValue = "".
            end.
            
            case tlvType:
                when {&COMMAND_PROGRAM} then
                    Command.Name = tlvValue.
                when {&COMMAND_PARAMETERS} then
                    Command.Parameters = tlvValue.
                when {&COMMAND_SCOPE} then
                    Command.Scope = tlvValue.
                when {&COMMAND_USES_LONGCHAR} then do:
                    Command.ResultIsLongChar = logical(tlvValue).
                end.
            end case.
        end.
        
    end.   
end.

/*
*  This is here since it is the only way to clear the return value
*  and the error status flag.
*/
procedure ClearReturnValue:
    return "".
end.

/*
*  This is here since it is the only way to clear the return value
*  and the error status flag.
*/
procedure HandleReply:
    define parameter buffer Command for ttCommand.
    define variable hdl as handle no-undo.
    hdl =  widget-handle(string(Command.RequestID)).
    if valid-handle(hdl) then 
    do:
        mResponse = Command.Parameters.
        apply "close" to hdl. 
    end.
end.

/*
This handles incoming commands from the Eclipse JVM.
*/
procedure ReceiveCommand:
    define buffer CommandBuffer for ttCommand.
    
    /* Read procedure for socket */
    define variable requestId   as integer   no-undo.
    define variable messageSize as integer   no-undo.
    define variable oneByte     as memptr    no-undo.
    define variable fourBytes   as memptr    no-undo.
    define variable theData     as memptr    no-undo.
    define variable cError      as character no-undo.
    define variable ok          as logical   no-undo.

    if not self:CONNECTED() then
    do:
        run QUIT in phserver ("Lost connection").
        return error "Socket disconnected".
    end.
    SET-SIZE(oneByte) = 1.
    SET-SIZE(fourBytes) = 4.
    
    /* Check if there is any data on the socket to read. Sometimes we get spurious socket responses.*/
    if self:GET-BYTES-AVAILABLE() = 0 then 
    return.
    
    self:READ(oneByte, 1, 1).
    if get-byte(oneByte, 1) = {&MSG_HEADER_BEGIN} then 
    do:
        self:READ(fourBytes, 1, 4).
        requestId = get-long(fourBytes, 1).
        if requestId > -1 then 
        do:
            self:READ(oneByte, 1, 1).
            if get-byte(oneByte, 1) = {&MSG_HEADER_END} then 
            do:
                self:READ(oneByte, 1, 1).
                if get-byte(oneByte, 1) = {&PACKET_BEGIN} then 
                do:
                    self:READ(fourBytes, 1, 4).
                    messageSize = get-long(fourBytes, 1).
                    if messageSize > -1 then 
                    do:
                        SET-SIZE(theData) = messageSize.
                        ok = self:READ(theData, 1, messageSize, read-exact-num).
                        if ok = true then 
                        do:
                            self:READ(oneByte, 1, 1).
                            if get-byte(oneByte, 1) = {&PACKET_END} then 
                            do:
                                self:READ(oneByte, 1, 1).
                                if get-byte(oneByte, 1) = {&MSG_END} then 
                                do:
                                end.
                                else
                                    cError = "Message End Marker".              
                            end.
                            else
                                cError = "Packet End Marker".
                        end.
                        else
                            cError = "Failed to read all of message data".
                    end.
                    else
                        cError = "Packet Size".
                end.
                else
                    cError = "Packet Begin".
            end.
            else
                cError = "Message Header End".
        end.
        else
            cError = "Request Id".
    end.
    else
        cError = "Message Header Begin".
    
    if cError = "" then 
    do:
        create CommandBuffer.
        CommandBuffer.requestID = requestID.
      
        run extractCommand(input theData, buffer CommandBuffer) no-error. 
        if (error-status:error) then 
            cError = "ERROR:" + error-status:get-message(1).
 
    end.      
      
    SET-SIZE(fourBytes) = 0.
    SET-SIZE(oneByte) = 0.
    SET-SIZE(theData) = 0.  
    
    if cError <> "" then 
    do:
        run QUIT in phserver("Error: " + cError).
        delete CommandBuffer.
        return error cError.
    end.
    else 
    do:
        if CommandBuffer.Scope = "REPLY" then
        do:
            log( "Reply " + string(CommandBuffer.requestid) + " : " + CommandBuffer.Parameters).
            run HandleReply(buffer CommandBuffer).
            delete CommandBuffer. 
        end.
        else do: 
            run executeCmd in phServer(buffer CommandBuffer) no-error.
            delete CommandBuffer. 
            if error-status:error or return-value <> "" then
            do:
                run QUIT in phServer("Error: " + return-value).
                return error return-value.
            end.
        end.
    end.  
  
end procedure.
 
