/* ***********************************************************/
/* Copyright (c) 2004-2016 by Progress Software Corporation  */
/*                                                           */
/* All rights reserved.  No part of this program or document */
/* may be  reproduced in  any form  or by  any means without */
/* permission in writing from Progress Software Corporation. */
/*************************************************************/
 
define new global shared variable OEIDE_Context    as handle    no-undo.
    
define variable forceQuit        as logical   no-undo.
define variable cPDir            as character no-undo.
 
define variable aOk              as logical   no-undo.
define variable fLoggerHandle    as handle    no-undo.
/* use variables for the common used servers - still started with startServer */
define variable fDefaultHandle   as handle    no-undo.
define variable fSocketClient    as handle    no-undo.
define variable fWait            as logical no-undo.
/* servers */ 
define temp-table ttServer no-undo
    field ServerHandle as handle
    field ServerName   as character
    index serverName as unique ServerName.
    
/* this temp-table only contains a single record and its buffer is used
   like an object to hold onto the options read from the command message.
*/   
define temp-table ttCommand no-undo
    field Name as character
    field Parameters as character
    field Scope as character
    field RequestID as int64
    field ResultIsLongChar as logical.

/* declaration for protocol */
&SCOPED-DEFINE MSG_HEADER_SIZE 7
&SCOPED-DEFINE PACKET_HEADER_SIZE 5
&SCOPED-DEFINE MSG_HEADER_BEGIN 254
&SCOPED-DEFINE MSG_HEADER_END   253
&SCOPED-DEFINE PACKET_BEGIN     252
&SCOPED-DEFINE PACKET_END       251
&SCOPED-DEFINE MSG_END          250

/* declarations for command identifiers */

/* the name of the program to run */
&SCOPED-DEFINE COMMAND_PROGRAM 1

/* the parameters to pass to the program that is being run */ 
&SCOPED-DEFINE COMMAND_PARAMETERS 2

/* the scope of the command.  Must be already registered */
&SCOPED-DEFINE COMMAND_SCOPE 3

/* use alternate run statement to allow longchar return value */
&SCOPED-DEFINE COMMAND_USES_LONGCHAR 4

function getServerHandle returns handle 
    (cServer as character) forward.
    

function startServer returns handle 
    (pcServer as character, pcProgramName as character ) forward.
 
function log returns character 
    ( input msg as character) forward.

/* forward declare context functions */ 
function getIsAppbuilderRunning returns logical 
    ( ) in OEIDE_context.

function getProjectName returns character 
    ( ) in OEIDE_context.

function getProjectFullPath returns character 
    ( ) in OEIDE_context.

function getProjectWorkDirectory returns character 
    ( ) in OEIDE_context.

function getServerDirectory returns character 
    () in OEIDE_context.

function setServerDirectory returns character 
    (pcDir as character) in OEIDE_context.
   
function setApplicationWait returns logical 
    (pIsWait as log) in OEIDE_context.    
    
/* set the handle of the socket client procedure that communicates with Eclipse */ 
function setSocketClient returns logical
        ( phClient as handle ) in OEIDE_context.
 
&global-define DEFAULT-SERVER-PORT 3333
&global-define MAX-CONNECTION-FAILURES 3

&IF '{&WINDOW-SYSTEM}' = 'GUI':U &THEN
  {src/adm2/globals.i}
&ENDIF

/******* Initialization ****/
cPDir = replace(program-name(1),"_server.p","").

/* Start the context object and set its server directory  */ 
run value(cpDir + "_oeidecontext.p") persistent set OEIDE_Context.

setServerDirectory(cPDir).
 
/* this will hide any session resets */
if session:display-type = 'GUI':U then
do:
    current-window:visible = false.
    current-window:title = "Progress". /* Forces realization of default window. */  
    session:appl-alert-boxes = true.    
    session:system-alert-boxes = true.  
end.

/* Start the context object and set its server directory  */ 
run value(cpDir + "_socketclient.p") persistent set fsocketClient (this-procedure)  .
 
if not valid-handle(fsocketClient) then
do: 
    run quit ("").
    quit.
end.
run setLoggerHandle in fSocketClient (fLoggerHandle) .
subscribe procedure fSocketClient to "setLoggerHandle" in this-procedure.   
setSocketClient(fSocketClient).

/* now that we have connected, wait forever listening for
   data from the server
   When we receive data, pass it to the read handler and then
   start looping again. 
*/

LOOPCHK:    
do while true:

    if(forceQuit) then 
    do:
       if fWait  then 
       do:
           System.Windows.Forms.Application:Exit().
       end.    
       quit.
    end.
    DONTQUIT:
    do on error     undo, retry DONTQUIT
        on stop      undo, retry DONTQUIT
        on endkey    undo, retry DONTQUIT
        on quit      undo, leave DONTQUIT: 
      &IF "{&WINDOW-SYSTEM}" = "TTY" &THEN 
          fWait = false.
          setApplicationWait(fWait).
          wait-for "CLOSE" of this-procedure.
      &ELSE
          /* not documented - backdoor just in case someone complains...  
             NOTE: This should cannot be used if VD is used as AppBuilder and VD  
                   will have nested wait stacks if used in same project and 
                   get error on close/restart (if closed in incorrect order, which 
                  seems to always be the case ) 
           */ 
         
          if os-getenv("PDS_WAITFOR") = "ABL":U  then
          do:
              fWait = false.
              setApplicationWait(fWait).
              wait-for "CLOSE":U of this-procedure.    
          end.
          else do:
              fWait = true.
              setApplicationWait(fWait).
              wait-for System.Windows.Forms.Application:run ().
          end.
      &ENDIF
    end.
    
end.

quit.

/* **********************  Internal Procedures  *********************** */
 
/* If a command was received from eclipse then process it here and
*  call the appropriate procedure.
*  Commands are sent in the format of type, length, command string 
*  followed by type, length, parameter string
*/

procedure executeCmd:
    define parameter buffer Command for ttCommand.
    
    define variable hTarget as handle    no-undo.
    define variable commandResult as longchar no-undo.
  
    run ClearReturnValue no-error.
    
    do on error undo, leave
        on stop undo, leave:

        if valid-handle(fLoggerHandle) then                                                
            log( "Request  " + string(Command.requestId) + ": " + Command.Scope + ":" + Command.Name + "(" + Command.Parameters + ")"). 
        if Command.Scope <> "EXTERNAL" then 
        do:
            case Command.Scope:
                when "DEFAULT" or when "UNDEFINED" then 
                    hTarget = fDefaultHandle.
                when "INTERNAL" then 
                    hTarget = this-procedure.
            end.
    
            if not valid-handle(hTarget) then  
            do:
                hTarget = getServerHandle(if Command.Scope = "UNDEFINED" then "default" else Command.Scope).
               
                if not valid-handle(hTarget) then 
                do:
                    commandResult = "ERROR:" + Command.Scope  + "ServerFailed".
                    undo, leave.
                end.
                
                case Command.Scope:
                    when "DEFAULT" or 
                    when "UNDEFINED" then 
                        fDefaultHandle = hTarget.
                end.      
            end.
        
            /* backward compatibility with internal entry check  */ 
            if Command.Scope = "UNDEFINED"  
            and not can-do(hTarget:internal-entries, Command.Name) then
                assign hTarget = ?
                       Command.Scope = "EXTERNAL".
                  
        end. /* not external */
    
        if Command.Scope = "EXTERNAL" then
        do: 
            FILE-INFORMATION:file-name = Command.Name.
            if FILE-INFORMATION:full-pathname = ? 
                or (index(FILE-INFORMATION:file-type, "F") = 0 and index(FILE-INFORMATION:file-type, "M") = 0) then 
                Command.Name = search(cPDir + "_ide" + Command.Name + ".p").
        end. 
    
        run ClearReturnValue no-error.
           
        if valid-handle(hTarget) then 
        do: 
            do on error  undo,leave on stop undo,leave on endkey undo,leave on quit undo,leave:
                if Command.ResultIsLongChar then do:
                    run value(Command.Name) in hTarget (Command.Parameters, output commandResult).
                end.
                else do:
                    run value(Command.Name) in hTarget (Command.Parameters).
                end.
            end.
            if not Command.ResultIsLongChar then            
                commandResult = return-value.      
        end.
        else if Command.Scope = "EXTERNAL" and Command.Name > "" then 
        do:
            do on error  undo,leave on stop undo,leave on endkey undo,leave on quit undo,leave:
                if Command.ResultIsLongChar then do:
                    run value(Command.Name)(Command.Parameters, output commandResult).
                end.
                else do:
                    run value(Command.Name)(Command.Parameters).
                end. 
            end.
            if not Command.ResultIsLongChar then
                commandResult = return-value.      
        end.
        else 
        do:
            commandResult = "ERROR:FileNotFound".
        end.
    end.
    
    if commandResult = "" and (error-status:error or error-status:get-message(1) > "") then 
        commandResult = "ERROR:" + error-status:get-message(1).
 
    if valid-handle(fLoggerHandle) then
    do:
        if length(commandResult) < 28000 then 
            log( "Response " + string(Command.requestid) + " : " + string(commandResult)).
        else if commandResult = ? then 
            log( "Response " + string(Command.requestid) + " : " +  "<value is unknown>").
        else   
            log( "Response " + string(Command.requestid) + " : " + "<value larger than 28K - " + string(length(commandResult)) + " >").
    end.
      
    run WriteToSocket in fSocketClient (Command.RequestId, commandResult) no-error.
  
    if error-status:error or return-value <> "" then
        return error return-value. 
end procedure.

/*
*  This is here since it is the only way to clear the return value
*  and the error status flag.
*/
procedure ClearReturnValue:
    return "".
end.

/*
*  This terminates the infinite wait-for "close"
*  and cleans up
*/
procedure QUIT:
    define input parameter cPrm as character no-undo.
    
    for each ttServer:
        /* Don't add no-error here.  add destroyobject to servers */ 
        if valid-handle(ttServer.ServerHandle) then
            run destroyObject in ttServer.ServerHandle.
        delete ttServer.    
    end.    

    delete object OEIDE_context no-error. 
    
    if valid-handle(fSocketClient) then 
    do:
        run Disconnect in fSocketClient.
        delete procedure fSocketClient.   
    end.
    if cPrm <> ? and length(cPrm) > 0 then
        log(cPrm).
    log("Quitting.").
    run CloseLogFile (input "").
    forceQuit = TRUE.
    QUIT.
    
end procedure.

/* ************************  Function Implementations ***************** */
function log returns character 
    ( input msg as character).
    
    if valid-handle(fLoggerHandle) then 
    do:
    
        dynamic-function("log" in fLoggerHandle, msg).
    end.
    
end function.

function getServerHandle returns handle
    ( pcServer as character ):

    /*------------------------------------------------------------------------------
            Purpose:                                                                      
            Notes:                                                                        
    ------------------------------------------------------------------------------*/    
    define variable hServer as handle no-undo.
    
    if not AVAILABLE ttServer or ttServer.ServerName <> pcServer then
    do:
        hServer = startServer(pcServer,"").       
    end.
    else 
        hServer = ttServer.ServerHandle.
   

    return hServer.   
        
end function.

function startServer returns handle 
    ( pcServer as character, pcProgramName as character ):
            /** this publish is only for UIB mode and retruns the handle for 
                adeuib/oeideuib.p -  */

    /*------------------------------------------------------------------------------
            Purpose:  Start a persistent server procedure (service)                                                                       
            Notes:  Returns the service if it already exists                                                                      
    ------------------------------------------------------------------------------*/    
    define variable hServer as handle no-undo.
    
    find ttServer where ttServer.ServerName = pcServer no-error.
    if not AVAILABLE ttServer then
    do:     
        /* allow adeuib code to start a server/service */
        if pcProgramName = " " then
        do:  
            publish "OEIDE_" + pcServer from this-procedure (output hServer).    
            /* allow the service to unregister */
                        
            if valid-handle(hServer) then
            do:
                subscribe procedure this-procedure to "OEIDE_UnregisterScope" in hServer 
                run-procedure "UnregisterScope".


            end.
        end.
        if not valid-handle(hServer) then
        do: 
            if pcProgramName = " " then
               pcProgramName = cPDir + "_server" + pcServer + ".p".
            if search(pcProgramName) = ? then leave. /* This is fix for remove the runtime error. We will remove this code later */
            do on stop undo, leave:
                run value (lc(pcProgramName)) persistent set hServer no-error.
            end.
        end.    
        if valid-handle(hServer) then
        do:
            run setContextHandle in hServer (OEIDE_Context) no-error.
            run setLoggerHandle in hServer (fLoggerHandle) no-error.
            if not error-status:error then
            do:
                subscribe procedure hServer to "setLoggerHandle" in this-procedure.        
            end.
            create ttServer.
            assign 
                ttServer.ServerName   = pcServer
                ttServer.ServerHandle = hServer.
        end.
    end.
    else 
        hServer = ttServer.ServerHandle.
        
    return hServer. 
end function.

/*
    start logging process and open the log file
*/
procedure OpenLogFile.

    define input parameter cPrm as character no-undo.
    
    define variable projectname as character no-undo.
   
    if valid-handle(fLoggerHandle) then 
    do:
        run CloseLogFile ("").
    end.
    run VALUE(cPDir + "_idelog.p") persistent set fLoggerHandle.
    run SetLogFile in fLoggerHandle (cPrm).
    run OpenLogFile in fLoggerHandle.
    publish "setLoggerHandle" from this-procedure(fLoggerHandle).
    log( "PATH=" + OS-GETENV("PATH")).
    log( "DLC=" + OS-GETENV("DLC")).
    log( "PROPATH=" + propath).
    log( "ECLIPSE_PROJECT=" + getProjectName()).
    log( "OEA_PORT=" + OS-GETENV("OEA_PORT")).
    log( "ECLIPSE_ROOT=" + OS-GETENV("ECLIPSE_ROOT")).
    log( "PDS_WAITFOR=" +  OS-GETENV("PDS_WAITFOR")).
    

end procedure.

procedure CloseLogFile.
    define input parameter cPrm as character no-undo.
    if valid-handle(fLoggerHandle) then 
    do:
        run CloseLogFile in fLoggerHandle.
        delete object fLoggerHandle.
        fLoggerHandle = ?.
        publish "setLoggerHandle" from this-procedure(fLoggerHandle).
    end.
end procedure.

/*------------------------------------------------------------------------------
        Purpose:  Start a persistent server procedure (service)                                                                       
        Notes:                                                                        
------------------------------------------------------------------------------*/
procedure RegisterScope.
    define input parameter cPrm as character no-undo.
    startServer(entry(1, cPrm, "|"),entry(2, cPrm, "|")).
    
end procedure.

/*------------------------------------------------------------------------------
        Purpose:  Stop and register a persistent command server (scope)                                                                       
        Notes:                                                                        
------------------------------------------------------------------------------*/
procedure UnregisterScope.
    define input parameter ServerName as character no-undo.
    find ttServer where ttServer.ServerName = ServerName no-error.
    if available ttServer then 
    do:
        if valid-handle(ttServer.ServerHandle) then 
        do:
            delete procedure ttServer.ServerHandle.
        end.
        delete ttServer.
    end.

end procedure.
