
/*------------------------------------------------------------------------
    File        : AddClassBindingToDIConfig.p
    Purpose     : 

    Syntax      :

    Description : 

    Author(s)   : Thiemann_M
    Created     : Thu May 16 09:13:47 CEST 2024
    Notes       :
  ----------------------------------------------------------------------*/

block-level on error undo, throw.

using Progress.Json.ObjectModel.JsonObject.
using Progress.Json.ObjectModel.JsonArray.
using Progress.Json.ObjectModel.ObjectModelParser.
using adm.method.cls.DMCOpSysSvc.

&SCOPED-DEFINE K_DICONFIG_SUBDIR  'DIConfig':U

function oGetBindingEntryOrNull returns JsonObject 
  (poBindingArray as JsonArray,
   pcBaseClass    as character) forward.

define input parameter gpcBaseClass   as character no-undo.
define input parameter gpcTargetClass as character no-undo.
define input parameter gpcCustomLevel as character no-undo.

run addClassToDIConfig(gpcBaseClass, gpcTargetClass, gpcCustomLevel).

/**
 * Adds binding for given base class to given source class.
 */


/* **********************  Internal Procedures  *********************** */


procedure addClassToDIConfig :
/**
 * Adds given class mapping to the di config files
 */

define input parameter pcBaseClass   as character no-undo.
define input parameter pcTargetClass as character no-undo.
define input parameter pcCustomLevel as character no-undo.

define variable cBaseClassFileName as character no-undo.
define variable cModule            as character  no-undo.
define variable cConfigFilename    as character  no-undo.

assign
  cBaseClassFileName = DMCOpSysSvc:cFileNameFromPath(pcBaseClass)
  cModule            = caps(substring(cBaseClassFileName, 1, 1))
  cConfigFilename    = (if pcCustomLevel > '':U then
                          substitute('DIConfig_&1_&2.json':U,
                                     cModule,
                                     pcCustomLevel)
                        else
                          substitute('DIConfig_&1.json':U,
                                     cModule))
  .

run addClassToRuntimeDIConfig(pcBaseClass, pcTargetClass, cConfigFilename).
run addClassToCodeDIConfig(pcBaseClass, pcTargetClass, cConfigFilename).

end procedure.


procedure addClassToRuntimeDIConfig :
/**
 * Adds given class to the di config file located in the runtime directories
 */

define input parameter pcBaseClass      as character no-undo.
define input parameter pcTargetClass    as character no-undo.
define input parameter pcConfigFilename as character no-undo.

define variable cConfigDir as character  no-undo.

cConfigDir = pACStartupSvc:cParameterValue('RessourceDir':U).

if not DMCOpSysSvc:lIsFileExistingSilent(cConfigDir, 'Dir':U) then
  return.

cConfigDir = DMCOpSysSvc:cConcatPath(cConfigDir, {&K_DICONFIG_SUBDIR}).

run addClassToDIConfigFile(pcBaseClass, pcTargetClass, cConfigDir, pcConfigFilename).

end procedure.


procedure addClassToCodeDIConfig :
/**
 * Adds given class to the di config file located in the code directories
 */

define variable cConfigDir         as character  no-undo.

define input parameter pcBaseClass      as character no-undo.
define input parameter pcTargetClass    as character no-undo.
define input parameter pcConfigFilename as character no-undo.

cConfigDir = pACStartupSvc:cParameterValue('WorkingDirectory':U).

if DMCOpSysSvc:cLastDirectoryFromPath(cConfigDir) <> 'src':U then
  return.

cConfigDir = DMCOpSysSvc:cConcatPath(DMCOpSysSvc:cDirectoryFromPath(cConfigDir), 'res':U).

if not DMCOpSysSvc:lIsFileExistingSilent(cConfigDir, 'Dir':U) then
  return.

cConfigDir = DMCOpSysSvc:cConcatPath(cConfigDir, {&K_DICONFIG_SUBDIR}).

run addClassToDIConfigFile(pcBaseClass, pcTargetClass, cConfigDir, pcConfigFilename).

end procedure.


procedure addClassToDIConfigFile :
/**
 * Adds given class mapping to given di config file
 */

define input parameter pcBaseClass      as character no-undo.
define input parameter pcTargetClass    as character no-undo.
define input parameter pcConfigDir      as character no-undo.
define input parameter pcConfigFilename as character no-undo.

define variable oConfigRoot        as JsonObject no-undo.
define variable oBindingsArray     as JsonArray  no-undo.
define variable oBinding           as JsonObject no-undo.

// If no config has been created yet, the config dir iteslf might still be
// missing as well.
DMCOpSysSvc:createDirectory(pcConfigDir).

assign
  pcConfigFilename = DMCOpSysSvc:cConcatPath(pcConfigDir, pcConfigFilename)
  oConfigRoot      = (if DMCOpSysSvc:lIsFileExistingSilent(pcConfigFilename, 'File':U) then
                        cast(new ObjectModelParser():ParseFile(pcConfigFilename), JsonObject)
                      else
                        new JsonObject()).
  .

if oConfigRoot:Has('bindings':U) then
  oBindingsArray = oConfigRoot:GetJsonArray('bindings':U).

else
do:
  oBindingsArray = new JsonArray().
  oConfigRoot:Add('bindings':U, oBindingsArray).
end.

oBinding = oGetBindingEntryOrNull(oBindingsArray, pcBaseClass).

if not valid-object(oBinding) then
do:
  oBinding = new JsonObject().
  oBindingsArray:Add(oBinding).
  oBinding:Add('objectType':U, pcBaseClass).
end.

if oBinding:Has('resolveTo':U) then
  oBinding:Set('resolveTo':U, pcTargetClass).
else
  oBinding:Add('resolveTo':U, pcTargetClass).

oConfigRoot:WriteFile(pcConfigFilename, yes).

end procedure.

function oGetBindingEntryOrNull returns JsonObject 
  ( poBindingArray as JsonArray,
    pcBaseClass    as character ):
/**
 * Returs the binding for given base class. Returns ?, if the binding does
 * not yet exist.
 */

define variable iBindingCounter as integer    no-undo.
define variable oBinding        as JsonObject no-undo.

do iBindingCounter = 1 to poBindingArray:length:
  
  oBinding = poBindingArray:GetJsonObject(iBindingCounter).
  
  if    oBinding:Has('objectType':U)
    and oBinding:GetCharacter('objectType':U) = pcBaseClass then
    return oBinding.
  
end.

return ?.

end function.
