
/*------------------------------------------------------------------------
    File        : get_unit_test_links.p
    Purpose     : 

    Syntax      :

    Description : 

    Author(s)   : Thiemann_M
    Created     : Sat Mar 25 07:03:35 CET 2023
    Notes       :
  ----------------------------------------------------------------------*/

/* ***************************  Definitions  ************************** */

block-level on error undo, throw.

using Progress.Lang.*.
using System.Text.RegularExpressions.*.

define input  parameter gpclRequest as longchar  no-undo.
define output parameter gopclResult as longchar  no-undo.

/* ********************  Preprocessor Definitions  ******************** */

&SCOPED-DEFINE K_UNIT_TEST_ANNOTATION_REGEX        'Test(?:\s*\(.*\))?':U

&SCOPED-DEFINE K_DELETE_INSTANCE_LIST_MATCHGROUP   ClassList
&SCOPED-DEFINE K_DELETE_INSTANCE_ANNOTATION_REGEX  'DeleteInstances\s*\(\s*classes=(?<{&K_CLASS_LIST_MATCHGROUP}>.*)\s*\)':U

/* Temp-Tables ---------------------------------------------------------------*/

/* Input Parameter / Request Temp-Tables */

define temp-table ttRequestParameters no-undo
  
  serialize-name 'Parameters':U
  
  field Id          as integer
    serialize-hidden
  field ListingsDir as character
  
  index Main is primary unique
    Id
  
  .

define temp-table ttRequestClass no-undo
  
  serialize-name 'Class':U
  
  field ClassName as character
    serialize-name 'Name':U
    xml-node-type 'attribute':U
  field ClassRelativeFilePath as character
    serialize-name 'RelativePath':U
  
  index Main is primary unique
    ClassName
  
  .

/* Output Parameter / Unit Test Class Temp-Tables */

define temp-table ttUnitTestClass
  
  serialize-name 'Class':U
  
  field ClassName             as character
    serialize-name 'Name':U
    xml-node-type 'attribute':U
  field FileModDate           as date
  field FileModTime           as integer
  field IsCompilable          as logical
  
  index Main is primary unique
    ClassName
  
  .

define temp-table ttUnitTestMethod no-undo
  
  serialize-name 'Method':U
  
  field ClassName             as character
    serialize-hidden
  field MethodName            as character
    serialize-name 'Name':U
    xml-node-type 'attribute':U
  field FileNumber            as integer
    serialize-hidden
  field RefSequenceStart      as integer
    serialize-hidden
  field RefSequenceEnd        as integer
    serialize-hidden
  field LineNumberStart       as integer
    serialize-hidden
  field LineNumberEnd         as integer
    serialize-hidden
  
  index Main is primary unique
    ClassName
    MethodName
  
  index LineNumber is unique
    ClassName
    FileNumber
    LineNumberStart
    LineNumberEnd
    MethodName
  
  .

define temp-table ttUnitTestMethodInvocation no-undo
  
  serialize-name 'Invocation':U
  
  field ClassName              as character
    serialize-hidden
  field MethodName             as character
    serialize-hidden
  field InvokedClassName       as character
    serialize-name 'InvokedClassName':U
  field InvokedMethodName      as character
    serialize-name 'InvokedMethodName':U
  field InvokedMethodSignature as character
    serialize-name 'InvokedMethodSignature':U
  
  index Main is primary unique
    ClassName
    MethodName
    InvokedClassName
    InvokedMethodName
    InvokedMethodSignature
  
  .

/*
define temp-table ttUnitTestMethodClassToDelete
  
  serialize-name 'DeleteClass':U
  
  field ClassName             as character
    serialize-hidden
  field MethodName            as character
    serialize-hidden
  field ClassToDelete         as character
    serialize-name 'Name':U
    xml-node-type 'attribute':U
  
  index Main is primary unique
    ClassName
    MethodName
    ClassToDelete
  
  .
*/

/* XRef Temp-Tables */

define /*{&ACCESS-LEVEL}*/ temp-table ttSource no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'Source':U
    field File-name   as character xml-node-type 'attribute':u 
    field Source-guid as character 
    field File-num    as integer
    
    index idx1 as primary unique Source-guid File-num
    index idx2                   File-name
    .

define /*{&ACCESS-LEVEL}*/ temp-table ttStringRef no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'String-ref':U
    field Source-guid   as character
    field Ref-seq       as integer 
    field Max-length    as integer
    field Justification as character
    field Translatable  as logical
    
    index idx1 as unique Source-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table ttParameterRef no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'Parameter-ref':U
    field Source-guid    as character
    field Ref-seq        as integer
    field Order          as integer   xml-node-type 'attribute':u 
    field Parameter-mode as character xml-node-type 'attribute':u 
    field Parameter-name as character xml-node-type 'attribute':u 
    field Parameter-type as character xml-node-type 'attribute':u 
    field Dimension      as integer
    field Is-append      as logical
    field Dataset-guid   as character
    
    index idx1 as unique Source-guid Ref-seq Order.

define /*{&ACCESS-LEVEL}*/ temp-table ttClassRef no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'Class-ref':U
    field Source-guid     as character
    field Ref-seq         as integer
    field Inherited-list  as character
    field Implements-list as character
    field Has-use-pool    as logical
    field Is-final        as logical
    field Is-serializable as logical
    field Dataset-guid    as character
    
    index idx1 as unique Source-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table ttReference no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'Reference':U
    field Object-identifier as character xml-node-type 'attribute':u 
    field Reference-type    as character xml-node-type 'attribute':u 
    field Source-guid       as character
    field File-num          as integer 
    field Ref-seq           as integer 
    field Line-num          as integer 
    field Object-context    as character 
    field Access-mode       as character 
    field Data-member-ref   as character 
    field Temp-ref          as character 
    field Detail            as character 
    field Is-static         as logical
    field Is-abstract       as logical
    
    index idx1 as primary unique
      Source-guid
      Ref-seq
      Line-num
    
    index RefSequence
      Ref-seq
    
    index RefTypeSequence
      Reference-type
      Ref-seq
    
    index FileRefSequence
      File-num
      Ref-seq
    
    index FileLineNumber
      File-num
      Line-num
      Ref-seq
    
    .

define /*{&ACCESS-LEVEL}*/ temp-table ttInterfaceRef no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'Interface-ref':U
    field Source-guid    as character
    field Ref-seq        as integer
    field Inherited-list as character
    
    index idx1 is unique primary Source-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table ttDatasetRef no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'Dataset-ref':U
    field Source-guid  as character
    field Dataset-guid as character
    field Ref-seq      as integer
    field N-uri        as character
    field N-prefix     as character
    field Is-reference as logical
    field Buffer-list  as character
    field Data-links   as integer
    
    index idx1 is unique primary Source-guid Dataset-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table ttRelation no-undo /*{&REFERENCE-ONLY}*/
    serialize-name 'Relation':U
    field Source-guid        as character
    field Dataset-guid       as character
    field Relation-name      as character xml-node-type 'attribute':u
    field Parent-buffer-name as character
    field Child-buffer-name  as character
    field Relation-list      as character
    
    index idx1 is primary Source-guid Dataset-guid.

/* Datasets ------------------------------------------------------------------*/

/* Input Parameter / Unit-Test Dataset Temp-Tables */

define dataset dsRequest
  
  serialize-name 'Request':U
  
  for ttRequestParameters,
      ttRequestClass
  
  .

/* Unit-Test Link Temp-Tables */

define dataset dsUnitTestXrefData
  
  serialize-name 'UnitTestXrefData':U
  
  for ttUnitTestClass,
      ttUnitTestMethod,
      ttUnitTestMethodInvocation
      /*
      ,
      ttUnitTestMethodClassToDelete
      */
  
  data-relation drTestMethod
    for ttUnitTestClass, ttUnitTestMethod
    relation-fields( ClassName, ClassName )
    nested
  
  data-relation drTestInvLnk
    for ttUnitTestMethod, ttUnitTestMethodInvocation
    relation-fields( ClassName, ClassName, MethodName, MethodName )
    nested
  
  /*
  data-relation drClassToDelete
    for ttUnitTestMethod, ttUnitTestMethodClassToDelete
    relation-fields( ClassName, ClassName, MethodName, MethodName )
    nested
  */
  .
  
/* XRef Temp-Dataset */

define /*{&ACCESS-LEVEL}*/ dataset dsXref /*{&REFERENCE-ONLY}*/
    namespace-uri 'uri:schemas-progress-com:XREFD:0005':U
    serialize-name 'Cross-reference':u 
    for ttSource, ttReference, ttClassRef, ttInterfaceRef, ttStringRef, ttParameterRef, ttDatasetRef, ttRelation
    data-relation for ttSource, ttReference
        relation-fields (Source-guid, Source-guid,File-num, File-num) 
        nested
    data-relation for ttReference, ttClassRef
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for ttReference, ttInterfaceRef
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for ttReference, ttStringRef
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for ttReference, ttParameterRef
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq) 
        nested
    data-relation for ttReference, ttDatasetRef
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for ttDatasetRef, ttRelation
        relation-fields (Source-guid, Source-guid,Dataset-guid, Dataset-guid)
        nested
    .

/* ***************************  Main Block  *************************** */

define variable goTestAnnotationRegex as Regex no-undo.

goTestAnnotationRegex = new Regex( {&K_UNIT_TEST_ANNOTATION_REGEX}, RegexOptions:IgnoreCase ).

dataset dsRequest:read-xml( 'LONGCHAR':U, gpclRequest, 'EMPTY':U, ?, ? ). 

find first ttRequestParameters
  no-error.

if not available ttRequestParameters then
  
  return.

run analyzeAllClasses( ttRequestParameters.ListingsDir ).

/* **********************  Internal Procedures  *********************** */

&IF DEFINED(EXCLUDE-addAllInvokationsToTempTable) = 0 &THEN

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE addAllInvokationsToTempTable Method-Library
procedure addAllInvokationsToTempTable :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcClassName as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable cInvokedClassName  as character no-undo.
define variable cInvokedMethodName as character no-undo.

/* Buffers -------------------------------------------------------------------*/

define buffer bttReference                for temp-table ttReference.

define buffer bttUnitTestMethod           for temp-table ttUnitTestMethod.
define buffer bttUnitTestMethodInvocation for temp-table ttUnitTestMethodInvocation.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

for each bttReference
  where bttReference.Reference-type = 'INVOKE':U
  use-index RefTypeSequence:
  
  if    bttReference.Object-identifier begins 'OpenEdge.':U
     or bttReference.Object-identifier begins 'Progress.':U
     or bttReference.Object-identifier begins 'tests.':U then
    
    next.
  
  find first bttUnitTestMethod
    where bttUnitTestMethod.ClassName       = pcClassName
      and bttUnitTestMethod.FileNumber      = bttReference.File-num
      and bttUnitTestMethod.LineNumberStart < bttReference.Line-num
      and bttUnitTestMethod.LineNumberEnd   > bttReference.Line-num
    no-error.
  
  if not available bttUnitTestMethod then
    
    next.
  
  assign
    cInvokedClassName  = entry( 1, bttReference.Object-identifier, ':':U )
    cInvokedClassName  = entry( num-entries( cInvokedClassName, '.':U ), cInvokedClassName, '.':U )
    cInvokedMethodName = entry( 2, bttReference.Object-identifier, ':':U )
    .
  
  if can-find( bttUnitTestMethodInvocation
                 where bttUnitTestMethodInvocation.ClassName              = bttUnitTestMethod.ClassName
                   and bttUnitTestMethodInvocation.MethodName             = bttUnitTestMethod.MethodName
                   and bttUnitTestMethodInvocation.InvokedClassName       = cInvokedClassName
                   and bttUnitTestMethodInvocation.InvokedMethodName      = cInvokedMethodName
                   and bttUnitTestMethodInvocation.InvokedMethodSignature = bttReference.Object-context ) then
    
    next.
  
  create bttUnitTestMethodInvocation.
  
  assign
    bttUnitTestMethodInvocation.ClassName              = bttUnitTestMethod.ClassName
    bttUnitTestMethodInvocation.MethodName             = bttUnitTestMethod.MethodName
    bttUnitTestMethodInvocation.InvokedClassName       = cInvokedClassName
    bttUnitTestMethodInvocation.InvokedMethodName      = cInvokedMethodName
    bttUnitTestMethodInvocation.InvokedMethodSignature = bttReference.Object-context
    .
  
  validate bttUnitTestMethodInvocation.
  
end.

end procedure. /* addAllInvokationsToTempTable */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

/* ************************  Function Prototypes ********************** */



function cFilenameBody returns character 
  (pcFilename as character) forward.


&ENDIF


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE addAllTestMethodsToTempTable Method-Library
procedure addAllTestMethodsToTempTable :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcClassName as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable iTestSequenceStart as integer   no-undo init ?.
define variable iTestLineStart     as integer   no-undo init ?.

/* Buffers -------------------------------------------------------------------*/

define buffer bttSource          for temp-table ttSource.
define buffer bttReference       for temp-table ttReference.

define buffer bttUnitTestMethod  for temp-table ttUnitTestMethod.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

for each bttReference
  use-index RefSequence:
    
  case bttReference.Reference-type:
    
    when 'ANNOTATION':U then
    do:
      
      if goTestAnnotationRegex:IsMatch( bttReference.Object-identifier ) then
        
        assign
          iTestSequenceStart = bttReference.Ref-seq
          iTestLineStart     = bttReference.Line-num
          .
      
    end.
    
    when 'METHOD':U then
    do:
      
      /* No test annotation is active. Therefore this method is not a testcase. */
      
      if iTestSequenceStart = ? then
        
        next.
      
      create bttUnitTestMethod.
      
      assign
        bttUnitTestMethod.ClassName        = pcClassName
        bttUnitTestMethod.MethodName       = bttReference.Object-identifier
        bttUnitTestMethod.FileNumber       = bttReference.File-num
        bttUnitTestMethod.RefSequenceStart = iTestSequenceStart
        bttUnitTestMethod.RefSequenceEnd   = bttReference.Ref-seq
        bttUnitTestMethod.LineNumberStart  = iTestLineStart
        bttUnitTestMethod.LineNumberend    = bttReference.Line-num
        .
      
      validate bttUnitTestMethod.
      
      assign
        iTestSequenceStart = ?
        iTestLineStart     = ?
        .
      
    end.
    
  end case.
  
end.

end procedure. /* addAllTestMethodsToTempTable */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE analyzeAllClasses Method-Library
procedure analyzeAllClasses :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcListingsDir as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable cUnitTestLinksCacheFile as character no-undo.

/* Buffers -------------------------------------------------------------------*/

define buffer bttRequestClass for temp-table ttRequestClass.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

assign
  pcListingsDir           = right-trim( pcListingsDir, '/\':U )
  cUnitTestLinksCacheFile = substitute( '&1\&2':U,
                                        pcListingsDir ,
                                        '_UnitTestLinks.xml':U ) 
  .

file-info:file-name = cUnitTestLinksCacheFile.

if file-info:full-pathname > '':U then
  
  dataset dsUnitTestXrefData:read-xml( 'FILE':U, cUnitTestLinksCacheFile, 'EMPTY':U, ?, ? ) no-error.

for each bttRequestClass:
  
  run analyzeSingleClass( pcListingsDir,
                          bttRequestClass.ClassName,
                          bttRequestClass.ClassRelativeFilePath ).
  
end.

dataset dsUnitTestXrefData:write-xml( 'LONGCHAR':U, gopclResult, no ).

copy-lob from gopclResult
  to file( cUnitTestLinksCacheFile ).

end procedure. /* analyzeAllClasses */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE analyzeSingleClass Method-Library
procedure analyzeSingleClass :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcListingsDir           as character no-undo.
define input parameter pcClassName             as character no-undo.
define input parameter pcClassRelativeFilePath as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable lIsCompilable  as logical   no-undo.
define variable cClassFilePath as character no-undo.
define variable cXrefFilename  as character no-undo.

/* Buffers -------------------------------------------------------------------*/

define buffer bttUnitTestClass for temp-table ttUnitTestClass.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

find bttUnitTestClass
  where bttUnitTestClass.ClassName = pcClassName
  no-error.

assign
  cClassFilePath      = search( pcClassRelativeFilePath ).
  file-info:file-name = cClassFilePath
  .

if available bttUnitTestClass then
do:
  
  /* Return if cache is already up to date. */
  
  if    bttUnitTestClass.FileModDate = file-info:file-mod-date
    and bttUnitTestClass.FileModTime = file-info:file-mod-time then
  
  return.

  /* Delete cached data that isn't up to date anymore. */
  
  run deleteDataForSingleClass( pcClassName ).
  
end.
 
/* Create and analyze xref information. */

cXrefFilename = substitute( '&1\&2.xref.xml':U,
                            pcListingsDir,
                            cFilenameBody( pcClassName ) ).

create bttUnitTestClass.

bttUnitTestClass.ClassName = pcClassName.

do on error undo, throw:
  
  compile value( search( cClassFilePath ) ) xref-xml value( cXrefFilename ).
  
  assign
    bttUnitTestClass.FileModDate  = file-info:file-mod-date
    bttUnitTestClass.FileModTime  = file-info:file-mod-time
    bttUnitTestClass.IsCompilable = yes.
    .
  
  catch oError as Error:
    
    assign
      bttUnitTestClass.FileModDate  = ?
      bttUnitTestClass.FileModTime  = 0
      bttUnitTestClass.IsCompilable = no
      .
    
    return.
    
  end catch.
  
end.

run analyzeXrefFile( pcClassName, pcClassRelativeFilePath, cXrefFilename ).

end procedure. /* analyzeSingleClass */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE analyzeXrefFile Method-Library
procedure analyzeXrefFile :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcClassName             as character no-undo.
define input parameter pcClassRelativeFilePath as character no-undo.
define input parameter pcXrefFilename          as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable lTest     as logical   no-undo.
define variable cTestCase as character no-undo.

/* Buffers -------------------------------------------------------------------*/

define buffer bttReference     for temp-table ttReference.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

file-info:file-name = pcClassRelativeFilePath.

dataset dsXref:read-xml( 'FILE':U, pcXrefFilename, 'EMPTY':U, ?, ? ).

run addAllTestMethodsToTempTable( pcClassName ).
run addAllInvokationsToTempTable( pcClassName ).

end procedure. /* analyzeXrefFile */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE deleteDataForSingleClass Method-Library
procedure deleteDataForSingleClass :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcClassName as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

/* Buffers -------------------------------------------------------------------*/

define buffer bttUnitTestClass            for temp-table ttUnitTestClass.
define buffer bttUnitTestMethod           for temp-table ttUnitTestMethod.
define buffer bttUnitTestMethodInvocation for temp-table ttUnitTestMethodInvocation.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

for each bttUnitTestMethodInvocation
  where bttUnitTestMethodInvocation.ClassName = pcClassName:
  
  delete bttUnitTestMethodInvocation.
  
end.

for each bttUnitTestMethod
  where bttUnitTestMethod.ClassName = pcClassName:
  
  delete bttUnitTestMethod.
  
end.

for each bttUnitTestClass
  where bttUnitTestClass.ClassName = pcClassName:
  
  delete bttUnitTestClass.
  
end.

end procedure. /* deleteDataForSingleClass */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


/* ************************  Function Implementations ***************** */

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION cFilenameBody Method-Library
function cFilenameBody returns character 
  ( pcFilename as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable cFileBody                    as character no-undo.
define variable iLastFilePathSeparataorIndex as integer   no-undo.
define variable iFileExtensionSeparatorIndex as integer   no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

iLastFilePathSeparataorIndex = max( r-index( pcFilename, '/':U ),
                                    r-index( pcFilename, '\':U ) ).

cFileBody = ( if iLastFilePathSeparataorIndex > 0 then
                substring( pcFilename, iLastFilePathSeparataorIndex + 1 )
              else
                pcFilename ).

iFileExtensionSeparatorIndex = r-index( cFileBody, '.':U ).

return ( if iFileExtensionSeparatorIndex > 0 then
           substring( cFileBody, 1, iFileExtensionSeparatorIndex - 1 )
         else
           cFileBody ).

end function. /* cFilenameBody */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME
