
/*------------------------------------------------------------------------
    File        : unittest_test.p
    Purpose     : 

    Syntax      :

    Description : 

    Author(s)   : Thiemann_M
    Created     : Mon Jan 23 12:33:53 CET 2023
    Notes       :
  ----------------------------------------------------------------------*/

/* ***************************  Definitions  ************************** */

block-level on error undo, throw.

using OpenEdge.ABLUnit.Runner.*.
using Progress.Json.ObjectModel.*.
using Progress.Lang.*.
using System.Text.RegularExpressions.*.

define input  parameter gpclTestCases  as longchar  no-undo.
define output parameter gopclResult    as longchar  no-undo.

/* ********************  Preprocessor Definitions  ******************** */

&SCOPED-DEFINE K_CLASS_LIST_MATCHGROUP    ClassList
&SCOPED-DEFINE K_CLASSES_TO_DELETE_REGEX  'DeleteInstances\s*\(\s*classes=(?<{&K_CLASS_LIST_MATCHGROUP}>.*)\s*\)':U

&SCOPED-DEFINE K_MESSAGE_TYPE_INFO        'Info':U
&SCOPED-DEFINE K_MESSAGE_TYPE_WARNING     'Warning':U
&SCOPED-DEFINE K_MESSAGE_TYPE_ERROR       'Error':U

&SCOPED-DEFINE K_RESULT_FILENAME          'results.xml':U

/* Temp-Tables ---------------------------------------------------------------*/

/* XRef Temp-Tables */

define /*{&ACCESS-LEVEL}*/ temp-table Source no-undo /*{&REFERENCE-ONLY}*/
    field File-name   as character xml-node-type 'attribute':u 
    field Source-guid as character 
    field File-num    as integer
    
    index idx1 as primary unique Source-guid File-num
    index idx2                   File-name
    .

define /*{&ACCESS-LEVEL}*/ temp-table String-ref no-undo /*{&REFERENCE-ONLY}*/
    field Source-guid   as character
    field Ref-seq       as integer 
    field Max-length    as integer
    field Justification as character
    field Translatable  as logical
    
    index idx1 as unique Source-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table Parameter-ref no-undo /*{&REFERENCE-ONLY}*/
    field Source-guid    as character
    field Ref-seq        as integer
    field Order          as integer   xml-node-type 'attribute':u 
    field Parameter-mode as character xml-node-type 'attribute':u 
    field Parameter-name as character xml-node-type 'attribute':u 
    field Parameter-type as character xml-node-type 'attribute':u 
    field Dimension      as integer
    field Is-append      as logical
    field Dataset-guid   as character
    
    index idx1 as unique Source-guid Ref-seq Order.

define /*{&ACCESS-LEVEL}*/ temp-table Class-ref no-undo /*{&REFERENCE-ONLY}*/
    field Source-guid     as character
    field Ref-seq         as integer
    field Inherited-list  as character
    field Implements-list as character
    field Has-use-pool    as logical
    field Is-final        as logical
    field Is-serializable as logical
    field Dataset-guid    as character
    
    index idx1 as unique Source-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table Reference no-undo /*{&REFERENCE-ONLY}*/
    field Object-identifier as character xml-node-type 'attribute':u 
    field Reference-type    as character xml-node-type 'attribute':u 
    field Source-guid       as character
    field File-num          as integer 
    field Ref-seq           as integer 
    field Line-num          as integer 
    field Object-context    as character 
    field Access-mode       as character 
    field Data-member-ref   as character 
    field Temp-ref          as character 
    field Detail            as character 
    field Is-static         as logical
    field Is-abstract       as logical
    
    index idx1 as primary unique Source-guid Ref-seq Line-num
    .

define /*{&ACCESS-LEVEL}*/ temp-table Interface-ref no-undo /*{&REFERENCE-ONLY}*/
    field Source-guid    as character
    field Ref-seq        as integer
    field Inherited-list as character
    
    index idx1 is unique primary Source-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table Dataset-ref no-undo /*{&REFERENCE-ONLY}*/
    field Source-guid  as character
    field Dataset-guid as character
    field Ref-seq      as integer
    field N-uri        as character
    field N-prefix     as character
    field Is-reference as logical
    field Buffer-list  as character
    field Data-links   as integer
    
    index idx1 is unique primary Source-guid Dataset-guid Ref-seq.

define /*{&ACCESS-LEVEL}*/ temp-table Relation no-undo /*{&REFERENCE-ONLY}*/
    field Source-guid        as character
    field Dataset-guid       as character
    field Relation-name      as character xml-node-type 'attribute':u
    field Parent-buffer-name as character
    field Child-buffer-name  as character
    field Relation-list      as character
    
    index idx1 is primary Source-guid Dataset-guid.

define temp-table ttClassToKill no-undo
  field ClassName as character
  index Main is primary unique
    ClassName
    .

/* Run Confg Temp-Tables */

define temp-table ttUnitTestClass no-undo
  serialize-name 'Class':U
  field ClassName as character
    serialize-name 'Name':U
    xml-node-type 'attribute':U
  field IsCompilable as logical
    serialize-hidden
  index Main is primary unique
    ClassName
    .

define temp-table ttUnitTestMethod no-undo
  serialize-name 'Method':U
  field ClassName as character
    serialize-hidden
  field MethodName as character
    serialize-name 'Name':U
    xml-node-type 'text':U
  index Main is primary unique
    ClassName
    MethodName
  .

/* Result Temp-Tables */

define temp-table ttTestResult no-undo
  serialize-name 'Result':U
  field ResultId   as integer
    serialize-hidden
  field ResultFile as character
  index Main is primary unique
    ResultId
  .

define temp-table ttTestMessage no-undo
  serialize-name 'Message':U
  field ResultId      as integer
    serialize-hidden
  field MessageId     as integer
    serialize-hidden
  field MessageType   as character
    serialize-name 'Type':U
    xml-node-type 'attribute':U
  field MessageText   as character
    serialize-name 'Text':U
  index Main is primary unique
    ResultId
    MessageId
    .

/* Datasets ------------------------------------------------------------------*/

/* XRef Temp-Dataset */

define /*{&ACCESS-LEVEL}*/ dataset dsXref /*{&REFERENCE-ONLY}*/
    namespace-uri 'uri:schemas-progress-com:XREFD:0005':U
    serialize-name 'Cross-reference':u 
    for Source, Reference, Class-ref, Interface-ref, String-ref, Parameter-ref, Dataset-ref, Relation
    data-relation for Source, Reference
        relation-fields (Source-guid, Source-guid,File-num, File-num) 
        nested
    data-relation for Reference, Class-ref
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for Reference, Interface-ref
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for Reference, String-ref
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for Reference, Parameter-ref
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq) 
        nested
    data-relation for Reference, Dataset-ref
        relation-fields (Source-guid, Source-guid,Ref-seq, Ref-seq)
        nested
    data-relation for Dataset-ref, Relation
        relation-fields (Source-guid, Source-guid,Dataset-guid, Dataset-guid)
        nested
    .

/* Run Confg Dataset */

define dataset dsRunConfig
  serialize-name 'Config':U
  for ttUnitTestClass,
      ttUnitTestMethod
  data-relation drMethod for ttUnitTestClass, ttUnitTestMethod
    relation-fields( ClassName, ClassName )
    nested
    .

/* Result Dataset */

define dataset dsResult
  serialize-hidden
  for ttTestResult,
      ttTestMessage
  .

/* ************************  Function Prototypes ********************** */

function oCreateOptionsJson returns JsonObject 
  (pcResultLocation as character) forward.

function oCreateOutputJson returns JsonObject 
  (pcResultLocation as character) forward.

function oCreateTestCasesJson returns JsonArray 
  (pcTestCases as character) forward.

function oCreateTestConfig returns TestConfig 
  ( pcResultLocation as character) forward.

function oCreateTestConfigJson returns JsonObject 
  (pcResultLocation as character) forward.

function oCreateTestJson returns JsonObject 
  (pcTest     as character,
   pcTestCase as character) forward.

function oCreateTestsJson returns JsonArray 
  () forward.

/* ***************************  Main Block  *************************** */

define variable gcResultDir            as character no-undo.
define variable gcResultFile           as character no-undo.
define variable goTestRunner           as ABLRunner no-undo.
define variable goClassesToDeleteRegex as Regex     no-undo.

define variable giMessageSequence      as integer   no-undo.
define variable giErrorMessageCounter  as integer   no-undo.

assign
  gcResultDir  = right-trim( session:temp-directory, '/\':U )
  gcResultFile = substitute( '&1\&2':U, gcResultDir, {&K_RESULT_FILENAME} )
  .

run addResult( gcResultFile ).

run deleteOldResultFile.

run prepareAllUnitTestClasses.

run killClassesFromTempTable.

goTestRunner = new ABLRunner( oCreateTestConfig( gcResultDir ),
                              'ABLUnitResult.log':U ).

run setSuppressAllMessages( yes ).

goTestRunner:RunTests().

catch goError as Error:
  
  do giErrorMessageCounter = 1 to goError:NumMessages:
    
    run addMessage( {&K_MESSAGE_TYPE_ERROR},
                               goError:GetMessage(giErrorMessageCounter)
                    + '~n':U
                    + '~n':U + goError:CallStack ).
    
  end.
  
end catch.

finally:
  
  // activate proALPHA Error Messages again
  
  run setSuppressAllMessages( no ).
  
  if valid-object( goTestRunner ) then
    
    delete object goTestRunner.
  
  dataset dsResult:write-xml( 'longchar':U, gopclResult, yes ).
  
end.

/* **********************  Internal Procedures  *********************** */


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE addClassesToKillFromDatasetToTempTable Method-Library
procedure addClassesToKillFromDatasetToTempTable :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oMatch            as class Match no-undo.

define variable cClassList        as character no-undo.
define variable iClassListCounter as integer   no-undo.
define variable cClassName        as character no-undo.

/* Buffers -------------------------------------------------------------------*/

define buffer bttClassToKill for temp-table ttClassToKill.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

for each Reference
  where Reference.Reference-type = "ANNOTATION":
  
  oMatch = goClassesToDeleteRegex:Match( Reference.Object-identifier ).
  
  cClassList = oMatch:Groups['{&K_CLASS_LIST_MATCHGROUP}':U]:value.
  
  do iClassListCounter = 1 to num-entries( cClassList ):
    
    cClassName = entry( iClassListCounter, cClassList ).
    
    if can-find( ttClassToKill
                   where ttClassToKill.ClassName = cClassName ) then
      
      return.
    
    create bttClassToKill.
    
    bttClassToKill.ClassName = cClassName.
    
    validate bttClassToKill.    
    
  end.
    
end.

end procedure. /* addClassesToKillFromDatasetToTempTable */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE addMessage Method-Library
procedure addMessage :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcMessageType as character no-undo.
define input parameter pcMessageText as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define buffer bttTestMessage for temp-table ttTestMessage.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

create bttTestMessage.

assign
  giMessageSequence          = giMessageSequence + 1
  bttTestMessage.MessageId   = giMessageSequence
  bttTestMessage.MessageType = pcMessageType
  bttTestMessage.MessageText = pcMessageText
  .

validate bttTestMessage.

end procedure. /* addMessage */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE addResult Method-Library
procedure addResult :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcResultFile as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define buffer bttTestResult for temp-table ttTestResult.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

create bttTestResult.

bttTestResult.ResultFile = pcResultFile.

validate bttTestResult.

end procedure. /* addResult */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE deleteOldResultFile Method-Library
procedure deleteOldResultFile :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

file-info:file-name = gcResultFile.

if file-info:full-pathname <> ? then
  
  os-delete value( gcResultFile ) no-error.

end procedure. /* deleteOldResultFile */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE fillXrefDatasetForSingleClass Method-Library
procedure fillXrefDatasetForSingleClass :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcClassName as character no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable cXrefFile            as character no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

cXrefFile = session:temp-dir + guid + '.xref':U.

do on error undo, throw:
  
  compile value( search( pcClassName ) ) xref-xml value( cXrefFile ).
  
  dataset dsXref:read-xml( 'file':u, search( cXrefFile ), 'EMPTY':U, ?, ? ).
  
  finally:
    os-delete value( cXrefFile ).
  end finally.
  
end.

end procedure. /* fillXrefDatasetForSingleClass */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE killInstancesFromAnnotations Method-Library
procedure killClassesFromTempTable :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oSessionObject        as Object    no-undo.
define variable oSessionObjectToCheck as Object    no-undo.

define variable cClassName            as character no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

if not temp-table ttClassToKill:has-records then
  
  return.

oSessionObject = session:first-object.

do while valid-object( oSessionObject )
  on error undo, throw:

  oSessionObjectToCheck = oSessionObject.

  // get next sibling here because of the guards
  oSessionObject = oSessionObject:next-sibling.

  if not valid-object( oSessionObjectToCheck:GetClass() ) then
    next.
  
  cClassName = oSessionObjectToCheck:GetClass():TypeName.
  
  if not can-find( ttClassToKill
                     where ttClassToKill.ClassName = cClassName ) then
    next.

  delete object oSessionObjectToCheck no-error.
  
end.

end procedure. /* killClassesFromTempTable */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE prepareAllUnitTestClasses Method-Library
procedure prepareAllUnitTestClasses :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define buffer bttUnitTestClass for temp-table ttUnitTestClass.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

goClassesToDeleteRegex = new Regex( {&K_CLASSES_TO_DELETE_REGEX} ).

dataset dsRunConfig:read-xml( 'longchar':U, gpclTestCases, 'EMPTY':U, ?, ? ).

for each bttUnitTestClass:
  
  run prepareSingleUnitTestClass( buffer bttUnitTestClass ).
  
end. /* for each bttUnitTestClass */

end procedure. /* prepareAllUnitTestClasses */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE prepareSingleUnitTestClass Method-Library
procedure prepareSingleUnitTestClass :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define parameter buffer pbttUnitTestClass for ttUnitTestClass.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

do on error undo, throw:
  
  run fillXrefDatasetForSingleClass( pbttUnitTestClass.ClassName ).
  
  catch oError as Error:
    return.
  end catch.
  
end.

pbttUnitTestClass.IsCompilable = yes.

run addClassesToKillFromDatasetToTempTable.

validate pbttUnitTestClass.

end procedure. /* prepareSingleUnitTestClass */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME


&ANALYZE-SUSPEND _UIB-CODE-BLOCK _PROCEDURE setSuppressAllMessages Method-Library
procedure setSuppressAllMessages :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter plSuppressAllMessages as logical   no-undo.

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oDMCMessageSvc as Object no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/

oDMCMessageSvc = dynamic-property( 'adm.method.cls.DMCMessageSvc':U, 'prpoInstance':U ) no-error.

if not valid-object( oDMCMessageSvc ) then
  
  return.

dynamic-property( oDMCMessageSvc, 'prplSuppressAllMessages' ) = plSuppressAllMessages.

end procedure. /* setSuppressAllMessages */


/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

/* ************************  Function Implementations ***************** */

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION oCreateOptionsJson Method-Library
function oCreateOptionsJson returns JsonObject 
  ( pcResultLocation as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oOptions as JsonObject no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/    

oOptions = new JsonObject().
oOptions:Add( 'output':U, oCreateOutputJson( pcResultLocation ) ).
oOptions:Add( 'throwError':U, true ).

return oOptions.

end function. /* oCreateOptionsJson */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION oCreateOutputJson Method-Library
function oCreateOutputJson returns JsonObject 
  ( pcResultLocation as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oOutput  as JsonObject no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/    

oOutput = new JsonObject().
oOutput:Add( 'location':U, pcResultLocation ).
oOutput:Add( 'format':U, 'xml':U ).

return oOutput.

end function. /* oCreateOutputJson */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION oCreateTestCasesJson Method-Library
function oCreateTestCasesJson returns JsonArray 
  ( pcTestCases as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oTestCases       as JsonArray no-undo.
define variable iTestCaseCounter as integer   no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/    

oTestCases = new JsonArray().
oTestCases:add( pcTestCases ).

return oTestCases.

end function. /* oCreateTestCasesJson */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION oCreateTestConfig Method-Library
function oCreateTestConfig returns TestConfig 
  ( pcResultLocation as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/    

return new TestConfig( oCreateTestConfigJson( pcResultLocation ) ).

end function. /* oCreateTestConfig */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION oCreateTestConfigJson Method-Library
function oCreateTestConfigJson returns JsonObject 
  ( pcResultLocation as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oTestConfigJson as JsonObject no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/    

/* Json structure example:
{
    "options": {
        "output": {
            "location": "results",
            "format": "xml"
        },
        "quitOnEnd": true,
        "writeLog": true,
        "showErrorMessage": true,
        "throwError": true
    },
    "tests": [
        {
            "test": "TestClass.cls",
            "cases": [
              "TestMethod1",
              "TestMethod2",
              "TestMethod3"
            ]
        }
    ]
}
*/

oTestConfigJson = new JsonObject().

oTestConfigJson:add( 'options':U, oCreateOptionsJson( pcResultLocation ) ).
oTestConfigJson:add( 'tests':U, oCreateTestsJson() ).

return oTestConfigJson.

end function. /* oCreateTestConfigJson */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION oCreateTestJson Method-Library
function oCreateTestJson returns JsonObject 
  ( pcTest     as character,
    pcTestCase as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oTest as JsonObject no-undo.

/* Buffers -------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/    

oTest = new JsonObject().

oTest:add( 'test':U, pcTest ).

if pcTestCase > '':U then
  
  oTest:add( 'cases':U, oCreateTestCasesJson( pcTestCase ) ).

return oTest.

end function. /* oCreateTestJson */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME

&ANALYZE-SUSPEND _UIB-CODE-BLOCK _FUNCTION oCreateTestsJson Method-Library
function oCreateTestsJson returns JsonArray 
  (  ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Notes ---------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* Parameters ----------------------------------------------------------------*/
/*                                                                            */
/* <none>                                                                     */
/*                                                                            */
/* Examples ------------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Variables -----------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/

define variable oTestsArray          as JsonArray no-undo.

define variable cTestEntry       as character no-undo.
define variable cTestName        as character no-undo.
define variable cTestCases       as character no-undo.
define variable iTestCaseCounter as integer   no-undo.

/* Buffers -------------------------------------------------------------------*/

define buffer bttUnitTestClass  for temp-table ttUnitTestClass.
define buffer bttUnitTestMethod for temp-table ttUnitTestMethod.

/*----------------------------------------------------------------------------*/
/* Processing                                                                 */
/*----------------------------------------------------------------------------*/    

oTestsArray = new JsonArray().

for each bttUnitTestClass:
  
  if not bttUnitTestClass.IsCompilable then
  do:
    
    run addMessage( {&K_MESSAGE_TYPE_WARNING},
                    substitute( 'UnitTest class &1 has not been run due to compile errors!':U,
                                bttUnitTestClass.ClassName ) ).
    
    next.
    
  end.
  
  if not can-find( first ttUnitTestMethod
                     where ttUnitTestMethod.ClassName = bttUnitTestClass.ClassName ) then
    
    oTestsArray:add( oCreateTestJson( search( bttUnitTestClass.ClassName ), ? ) ).
  
  else
    
    /* Normally we would just create an array entry for each test case. This  */
    /* somehow does not work and only the first test case entry entry will be */
    /* run. Therefore we create a separate test for each test case.           */
    
    for each bttUnitTestMethod
      where bttUnitTestMethod.ClassName = bttUnitTestClass.ClassName:
    
      oTestsArray:add( oCreateTestJson( search( bttUnitTestClass.ClassName ), bttUnitTestMethod.MethodName ) ).
      
    end.
  
end.

return oTestsArray.

end function. /* oCreateTestsJson */

/* _UIB-CODE-BLOCK-END */
&ANALYZE-RESUME