
/*------------------------------------------------------------------------
    File        : test3.p
    Purpose     : 

    Syntax      :

    Description : 

    Author(s)   : Thiemann_M
    Created     : Wed Dec 14 10:26:35 CET 2022
    Notes       :
  ----------------------------------------------------------------------*/

/* ***************************  Definitions  ************************** */

block-level on error undo, throw.

using System.Text.RegularExpressions.*.
using Progress.Json.ObjectModel.*.
using adm.method.cls.*.
using System.*.
using Progress.Lang.*.

define input parameter gpcInputFile as character no-undo.

/* ********************  Preprocessor Definitions  ******************** */

&GLOBAL-DEFINE pa-XBasisName          import_additional_db_field_p
&SCOPED-DEFINE ppDatasetName          dsDR_AddUIFields

&SCOPED-DEFINE K_COLOR_DARK_GREEN     2
&SCOPED-DEFINE K_COLOR_DARK_GREY      7
&SCOPED-DEFINE K_COLOR_RED            12

&SCOPED-DEFINE K_MISSING_PLACEHOLDER  '<<<Missing>>>':U

&SCOPED-DEFINE K_INFO_FIELD_SUFFIX    '_Info':U

/* ***************************  Definitons  *************************** */

define variable ghDummyWindow             as handle    no-undo.
define variable gcTitlePrefix             as character no-undo.

define variable gcCustomLevel             as character no-undo.

define variable gcDRC_Instance_ID         as character no-undo.
define variable gcDRC_Instance_Obj        as character no-undo.

define variable gclFileContent            as longchar  no-undo.

define variable goIncludeRegex            as Regex     no-undo extent 2.
define variable goIncludeParameterRegex   as Regex     no-undo extent 9.
define variable goReadOnlyRegex           as Regex     no-undo.
define variable goCoreNameRegex           as Regex     no-undo.
define variable goTableInfoFieldRegex     as Regex     no-undo.
define variable goLineNoRegex             as Regex     no-undo.

define variable giFieldSequence           as integer   no-undo.

define temp-table ttField no-undo
  field Id                   as integer
  field OriginalName         as character
    format 'x(128)':U
    column-label 'Original Name':U
  field TargetName           as character
    format 'x(32)':U
    column-label 'Target Name':U
  field TableName            as character
    format 'x(25)':U
    column-label 'Table':U
  field IsValidTable         as logical
  field FieldName            as character
    format 'x(32)':U
    column-label 'Field':U
  field IsValidField         as logical
  field DomainName           as character
    format 'x(40)':U
    column-label 'Domain':U
  field IsValidDomain        as logical
  field IsReadOnly           as logical
    column-label 'Read-Only':U
    view-as toggle-box
  field IsAddDBFieldExisting as logical
  field CanImport            as logical
  field IsSelected           as logical
    column-label 'Sel':U
    view-as toggle-box
  field IsInfofield          as logical
    column-label 'Info Field':U
    view-as toggle-box 
  field StartOffset          as integer
  field EndOffset            as integer
  field LineNo               as integer
    format 'z,zzz,zzz,zz9 ':U
    column-label 'Line':U
  index Main is primary unique
    Id
    .

{adm/repos/incl/dr_aui00.pds &ippNoReferenceOnlySwitch = "yes"}

/* ************************  Function Prototypes ********************** */


function clInsertText returns LONGCHAR 
  (pclText         as longchar,
   pclTextToInsert as longchar,
   piInsertOffset  as integer) forward.


function cPromptForDomainOrNull returns character 
  (pcDomain as character) forward.



function cPromptForTargetNameOrNull returns character 
  (pcTargetName as character) forward.


function cStripNamePrefix returns character 
  (pcName as character) forward.


function cTargetName returns character 
  (pcOriginalName as character) forward.


function iGetLineNumberOfOffset returns INTEGER 
  (pclText  as longchar,
   piOffset as integer) forward.


function iNextFieldOrderNumber returns INTEGER 
  (  ) forward.


function lCanImport returns logical 
  (buffer pbttField for ttField) forward.


function lIsValidDomain returns LOGICAL 
  (pcDomain as character) forward.


function lIsValidField returns logical 
  (pcTable as character,
   pcField as character) forward.


function lIsValidTable returns LOGICAL 
  (pcTable as character) forward.


function lPromptForFieldSelection returns logical 
  (  ) forward.


function oExtractTableFieldInfoFromInfoVariableName returns JsonObject 
  (pcFieldName as character) forward.


function oPromptForFieldOrNull returns JsonObject 
  (pcTable as character,
   pcField as character) forward.


function oVariableDefinitionInfo returns JsonObject 
  (pcOriginalName as character) forward.


/* ***************************  Main Block  *************************** */

do on error undo, throw:
  
  
  run initialize( gpcInputFile ).
  
  run importFromFile( gpcInputFile ).
  
  catch oError as Error :
    run showError( oError ).
  end catch.
  
  finally:
    delete object ghDummyWindow no-error.
  end finally.
  
end.

/* **********************  Internal Procedures  *********************** */

procedure addAllFieldsFromFileToSelectionTT:
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable iRegexCounter   as integer   no-undo.
define variable oIncludeMatches as MatchCollection no-undo.
define variable iMatchCounter   as integer   no-undo.

do iRegexCounter = 1 to extent( goIncludeRegex ):
  
  oIncludeMatches = goIncludeRegex[iRegexCounter]:Matches( gclFileContent ).
  
  do iMatchCounter = 0 to oIncludeMatches:count - 1:
    
    run addAllFieldsFromIncludeReferenceToSeledctionTT( oIncludeMatches[iMatchCounter]:Groups['IncludeName':U]:Value,
                                                        oIncludeMatches[iMatchCounter]:Groups['IncludeReferenceContent':U]:Value,
                                                        oIncludeMatches[iMatchCounter]:Groups['IncludeReferenceContent':U]:Index ).
    
  end.
  
end.

end procedure. /* addAllFieldsFromFileToSelectionTT */

procedure addAllFieldsFromIncludeReferenceToSeledctionTT :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcIncludeName                   as character no-undo.
define input parameter pclIncludeReferenceContent      as longchar  no-undo.
define input parameter piIncludeReferenceContentOffset as integer no-undo.

define variable oReadOnlyMatch   as Match           no-undo.
define variable lReadOnly        as logical         no-undo.

define variable iFieldCounter    as integer         no-undo.
define variable oContentMatches  as MatchCollection no-undo.
define variable iMatchCounter    as integer         no-undo.

define variable iStartOffset     as integer   no-undo.
define variable iEndOffset       as integer   no-undo.

define variable oParameterValues as JsonObject      no-undo.

assign
  oReadOnlyMatch = goReadOnlyRegex:Match( pclIncludeReferenceContent )
  lReadOnly      = (     oReadOnlyMatch:Success
                     and logical( oReadOnlyMatch:Groups['ReadOnly':U]:Value)  )
  .

do iFieldCounter = 1 to 9:
  
  oContentMatches  = goIncludeParameterRegex[iFieldCounter]:Matches(pclIncludeReferenceContent).
  
  if oContentMatches:count = 0 then
    next.
  
  assign
    oParameterValues = new JsonObject()
    iStartOffset     = 0
    iEndOffset       = 0
    .
  
  do iMatchCounter = 0 to oContentMatches:Count - 1:
    
    if iMatchCounter = 0 then
      iStartOffset = oContentMatches[iMatchCounter]:Index + 1.
     
    oParameterValues:Add(oContentMatches[iMatchCounter]:Groups['ParameterName':U]:Value,
                         oContentMatches[iMatchCounter]:Groups['ParameterValue':U]:Value).
    
    iEndOffset =   oContentMatches[iMatchCounter]:Index
                 + oContentMatches[iMatchCounter]:Length + 1.
    
  end.
  
  run addSingleFieldToSelectionTT( pcIncludeName,
                                   oParameterValues,
                                   lReadOnly,
                                   piIncludeReferenceContentOffset + iStartOffset,
                                   piIncludeReferenceContentOffset + iEndOffset ).
  
end.

end procedure. /* addAllFieldsFromIncludeReferenceToSeledctionTT */


procedure addSingleFieldToSelectionTT :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcIncludeName      as character  no-undo.
define input parameter poParameterValues  as JsonObject no-undo.
define input parameter plReadOnly         as logical    no-undo.
define input parameter piStartOffset      as integer    no-undo.
define input parameter piEndOffset        as integer    no-undo.

define buffer bttField for temp-table ttField.

create bttField.

assign
  giFieldSequence               = giFieldSequence + 1
  bttField.Id                   = giFieldSequence
  bttField.IsReadOnly           = plReadOnly
  bttField.StartOffset          = piStartOffset
  bttField.EndOffset            = piEndOffset
  bttField.LineNo               = iGetLineNumberOfOffset( gclFileContent, bttField.StartOffset )
  .

run initializeFieldFromIncludeReferenceParameters( pcIncludeName,
                                                   poParameterValues,
                                                   buffer bttField ).

/* TODO: Strip comments instead? */
if bttField.OriginalName begins '<<<':U then
do:
  delete bttField.
  return.
end.

run initializeFieldFromExistingRepositoryRecord( buffer bttField ).

run initizalieMissingFieldSourceProperties( buffer bttField ).

bttField.CanImport = lCanImport( buffer bttField ).

validate bttField.

end procedure. /* addSingleFieldToSelectionTT */


procedure commentOutSelectedFieldsInContent :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define buffer bttField for temp-table ttField.

define variable iIndentation as integer   no-undo.

for each bttField
  where bttField.IsSelected = yes
  by bttField.Id descending:
  
  assign
    iIndentation   = bttField.StartOffset - r-index( gclFileContent, '~n':U, bttField.StartOffset ) - 1 
    gclFileContent = clInsertText( gclFileContent,
                                   substitute( '~n&2<-- Remove vom sourcecode: imported as Additional UI Field &1 */':U,
                                               quoter( bttField.TargetName ),
                                               fill( ' ':U, iIndentation ) ),
                                   bttField.EndOffset )
    gclFileContent = clInsertText( gclFileContent,
                                   substitute( '/* TODO: Remove vom sourcecode: imported as Additional UI Field &1 -->~n&2':U,
                                               quoter( bttField.TargetName ),
                                               fill( ' ':U, iIndentation ) ),
                                   bttField.StartOffset )
    .
  
end.

end procedure. /* commentOutSelectedFieldsInContent */


procedure createDummyWindow :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

/* Create hidden dummy window to prevent the progress default window to be    */
/* shown alongside dialogs.                                                   */
create window ghDummyWindow.


assign
  ghDummyWindow:visible       = no
  ghDummyWindow:always-on-top = yes
  /* We center the window on the screen since the default location is kind of */
  /* random and the dialogs that use this window as parent will be shown      */
  /* centered to this window.                                                 */
  ghDummyWindow:x = ( session:width-pixels - ghDummyWindow:width-pixels ) / 2
  ghDummyWindow:y = ( session:height-pixels - ghDummyWindow:height-pixels ) / 2
  .

end procedure. /* createDummyWindow */


procedure createSelectedFieldsInRepository :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable iNextFieldOrderNo    as integer   no-undo.

define variable cDomainRef_Obj       as character no-undo.

define variable hTableRefbufferField as handle    no-undo.

define buffer bttField          for temp-table ttField.
define buffer bttDRC_AddUIField for temp-table ttDRC_AddUIField.

{call
  {&ppDatasetName}
  modifyTrackingChanges
  yes}.

iNextFieldOrderNo = iNextFieldOrderNumber().

hTableRefbufferField = buffer bttDRC_AddUIField:buffer-field( 'TableRef_Obj':U ) no-error.

for each bttField
  where bttField.IsSelected = yes:
  
  if bttField.FieldName > '':U then
    
    cDomainRef_Obj = ( if {fnarg
                            pa_lReposIsAddDBFieldExisting
                            "bttField.TableName,
                             bttField.FieldName"} then
                         {fnarg
                           pa_cReposVirtualFieldInformationByName
                           "bttField.TableName,
                            bttField.FieldName,
                            'DRC_VirtualField_Obj':U"}
                       else
                         {fnarg
                           pa_cReposFieldInformationByName
                           "bttField.TableName,
                            bttField.FieldName,
                            'DRC_Field_Obj':U"} ).
    
  else
    
    cDomainRef_Obj = {fnarg
                           pa_cReposDomainInformationByName
                           "bttField.DomainName,
                            'DRC_Domain_Obj':U"}.
  
  create bttDRC_AddUIField.
  
  assign
    bttDRC_AddUIField.DRC_Instance_Obj   = gcDRC_Instance_Obj
    bttDRC_AddUIField.DRC_AddUIField_Obj = DMCSessionSvc:cObjectID( 'DRC_AddUIField':U )
    bttDRC_AddUIField.DRC_AddUIField_ID  = bttField.TargetName
    hTableRefbufferField:buffer-value    = {fnarg
                                             pa_cReposTableInformationByName
                                             "bttField.TableName,
                                              'DRC_Table_Obj':U"}
        when valid-handle( hTableRefbufferField )
    bttDRC_AddUIField.DomainRef_Obj      = cDomainRef_Obj
    bttDRC_AddUIField.IsRange            = no
    bttDRC_AddUIField.ReadOnly           = bttField.IsReadOnly
    bttDRC_AddUIField.FieldOrderNumber   = iNextFieldOrderNo
    .
  
  validate bttDRC_AddUIField.
  
  iNextFieldOrderNo = iNextFieldOrderNo + 10.
  
end.

end procedure. /* createSelectedFieldsInRepository */


procedure fillDataset :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcDRC_Instance_Obj as character no-undo.

{call
  {&ppDatasetName}
  modifyTrackingChanges
  no}.

{call
  {&ppDatasetName}
  emptyDataset}.

adm.method.cls.DMCSessionSvc:setObjectProperty
  ( dataset {&ppDatasetName}:handle,
    'ttDRC_InstanceAddPrepareString':U,
    substitute(   'and DRC_Instance.DRC_Instance_Obj = &1':U,
                quoter( pcDRC_Instance_Obj ) ) ).

{call
  {&ppDatasetName}
  fillDataset}.

end procedure. /* fillDataset */


procedure importFromFile :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcFilename as character no-undo.

copy-lob from file pcFilename to gclFileContent.

run fillDataset( gcDRC_Instance_Obj ).

run addAllFieldsFromFileToSelectionTT.

if not temp-table ttField:has-records then
do:
  
  message
    'There is nothing to import!':U skip
    view-as alert-box
    warning
    buttons ok
    in window ghDummyWindow.
  
  return.
  
end.

if not lPromptForFieldSelection() = yes then
  return.

if not can-find( first ttField
                   where ttField.IsSelected = yes ) then
do:
  
  message
    'No Fields have been selected':U skip
    view-as alert-box
    warning
    buttons ok
    in window ghDummyWindow.
  
  return.
  
end.

run createSelectedFieldsInRepository.

{call
  {&ppDatasetName}
  saveChanges}.

run commentOutSelectedFieldsInContent.

copy-lob from gclFileContent to file pcFilename.

message
  'The selected fields have been imported':U skip
  view-as alert-box
  information
  buttons ok
  in window ghDummyWindow.

end procedure. /* importFromFile */


procedure initialize :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcInputFile as character no-undo.

define variable iFieldCounter as integer   no-undo.

gcTitlePrefix = ( if pACConnectionSvc:prpcSessionTitle <> ? then
                    pACConnectionSvc:prpcSessionTitle + ' - ':U
                  else
                    '':U ).

run createDummyWindow.

if    not pcInputFile matches '*~.w':U
  and not pcInputFile matches '*~..xw':U then
  undo, throw new AppError( 'Only ui programs are supported!':U, -1 ).

gcDRC_Instance_ID = DMCOpSysSvc:cFileBodyFromPath( pcInputFile ) + '.w':U.

if not {fnarg
         pa_lReposIsInstanceExisting
         "gcDRC_Instance_ID"} then
  
  undo, throw new AppError( substitute( 'Instance &1 is not defined in repository!':U,
                                        quoter( gcDRC_Instance_ID, '~'':U ) ),
                            -1 ).

gcDRC_Instance_Obj = {fnarg
                       pa_cReposInstanceInformationByName
                       "gcDRC_Instance_ID,
                        'DRC_Instance_Obj':U"}.

assign
  gcCustomLevel     = pACStartupSvc:cParameterValue( 'CustomLevel':U )
  goIncludeRegex[1] = new Regex( '~{adm\/template\/incl\/(?<IncludeName>dt_ind0[12]\.if)(?<IncludeReferenceContent>[^}]*)}':U,
                                 RegexOptions:IgnoreCase )
  goIncludeRegex[2] = new Regex( '~{x\/adm\/incl\/(?<IncludeName>xd_ind01\.if)(?<IncludeReferenceContent>[^}]*)}':U,
                                 RegexOptions:IgnoreCase )
  goReadOnlyRegex   = new Regex( '&ccNoUpdate\s*=\s*"(?<ReadOnly>[^"]*)"':U,
                                 RegexOptions:IgnoreCase )
  .

do iFieldCounter = 1 to 9:
  goIncludeParameterRegex[iFieldCounter] = new Regex( substitute( '&&(?<ParameterName>\S+)&1\s*=\s*"(?<ParameterValue>[^"]*)"':U,
                                                                  string( iFieldCounter ) ),
                                                      RegexOptions:IgnoreCase ).
end.

assign
  goCoreNameRegex       = new Regex( '^(?:[xXyY]g?(?:[cdilt]|cl|tz)?|g?(?:[cdilt]|cl|tz)?[xXyY])(?<CoreName>\S+)':U )
  goTableInfoFieldRegex = new Regex( '(?<TableName>[^\s_]~{1,3}_[^\s_]+)_(?<FieldName>\S+)':U + {&K_INFO_FIELD_SUFFIX}, RegexOptions:IgnoreCase )
  goLineNoRegex         = new Regex( '\n':U )
  .

end procedure. /* initialize */


procedure initializeFieldFromExistingRepositoryRecord :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define parameter buffer pbttField for temp-table ttField.

define buffer bttDRC_AddUIField for temp-table ttDRC_AddUIField.

find bttDRC_AddUIField
  where bttDRC_AddUIField.DRC_Instance_Obj  = gcDRC_Instance_Obj
    and bttDRC_AddUIField.DRC_AddUIField_ID = pbttField.TargetName
  no-error.

if available bttDRC_AddUIField then
do:
  
  pbttField.IsAddDBFieldExisting = yes.
  
  case DMCSessionSvc:cOwningTable( bttDRC_AddUIField.DomainRef_Obj ):
    
    when 'DRC_Domain':U then
      
      assign
        pbttField.DomainName    = {fnarg
                                    pa_cReposDomainInformationByObj
                                    "bttDRC_AddUIField.DomainRef_Obj,
                                     'DRC_Domain_ID':U"}
        pbttField.IsValidDomain = yes
        .
    
    when 'DRC_Field':U then
      
      assign
        pbttField.TableName      = {fnarg
                                     pa_cReposFieldInformationByObj
                                     "bttDRC_AddUIField.DomainRef_Obj,
                                      'DRC_Table_ID':U"}
        pbttField.IsValidTable   = yes
        pbttField.FieldName      = {fnarg
                                     pa_cReposFieldInformationByObj
                                     "bttDRC_AddUIField.DomainRef_Obj,
                                      'DRC_Field_ID':U"}
        pbttField.IsValidField   = yes
        .
    
  end.
  
end.


end procedure. /* initializeFieldFromExistingRepositoryRecord */


procedure initializeFieldFromIncludeReferenceParameters :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter pcIncludeName     as character  no-undo.
define input parameter poParameterValues as JsonObject no-undo.

define parameter buffer pbttField for temp-table ttField.

define variable cOriginalName as character  no-undo.
define variable cTable        as character  no-undo.
define variable cField        as character  no-undo.

case pcIncludeName:
  
  when 'dt_ind01.if':U then
  do:
    
    if not poParameterValues:Has('Field':U) then
      return.
    
    cOriginalName = poParameterValues:GetCharacter('Field':U).
    
    if num-entries( cOriginalName, '.':U ) >= 2 then
      
      assign
        cTable = entry( 1, cOriginalName, '.':U )
        cField = entry( 2, cOriginalName, '.':U )
        .
    
  end.
  
  otherwise
  do:
    
    if    not poParameterValues:Has( 'Buffer':U )
       or not poParameterValues:Has( 'FieldName':U ) then
      return.
    
    assign
      cTable        = poParameterValues:GetCharacter('Buffer':U)
      cField        = poParameterValues:GetCharacter('FieldName':U)
      cOriginalName = ( if cTable > '':U then
                          substitute( '&1.&2':U, cTable, cField )
                        else
                          cField ). 
      .
    
    if lIsValidTable( cTable ) then
      
      cTable = {fnarg
                 pa_cReposTableInformationByName
                 "cTable,
                  'DRC_Table_ID':U"}.
    
  end.
  
end.

assign
  pbttField.OriginalName = cOriginalName
  pbttField.IsInfoField  = pbttField.OriginalName matches '*':U + {&K_INFO_FIELD_SUFFIX}
  pbttField.TargetName   = ( if pbttField.IsInfoField then
                               pbttField.OriginalName
                             else
                               cTargetName( if cField > '':U then
                                              cField
                                            else
                                              replace( cOriginalName, '.':U, '_':U ) ) )
  pbttField.TableName    = cTable
  pbttField.IsValidTable = lIsValidTable( cTable )
      when cTable > '':U
  pbttField.FieldName    = cField
  pbttField.IsValidField = lIsValidField( cTable, cField )
      when cField > '':U
  .

end procedure. /* initializeFieldFromIncludeReferenceParameters */


procedure initializeMissingInfoFieldSourceProperties :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter poVariableInfo as JsonObject no-undo.

define parameter buffer pbttField for temp-table ttField.

define variable cCoreName             as character  no-undo.
define variable oSourceTableFieldInfo as JsonObject no-undo.
define variable oSourceVariableInfo   as JsonObject no-undo.
define variable cTableFromName        as character  no-undo.
define variable cFieldFromName        as character  no-undo.

oSourceVariableInfo = oVariableDefinitionInfo(substring( pbttField.OriginalName,
                                                         1,
                                                         length( pbttField.OriginalName ) - length( {&K_INFO_FIELD_SUFFIX} ) ) ).

if    oSourceVariableInfo:Has( 'DataType':U )
  and oSourceVariableInfo:GetCharacter( 'DataType':U ) begins 'Date':U then
do:
  
  assign
    pbttField.DomainName    = DMCAdditionalUIFieldsSvc:cInfoDomainIDForDateField()
    pbttField.IsValidDomain = lIsValidDomain( pbttField.DomainName )
    .
  
  return.
  
end.

oSourceTableFieldInfo = oExtractTableFieldInfoFromInfoVariableName( cStripNamePrefix( pbttField.OriginalName ) ).

if    oSourceTableFieldInfo:Has( 'DataType':U )
  and oSourceTableFieldInfo:GetCharacter( 'DataType':U ) begins 'Date':U then
do:
  
  assign
    pbttField.DomainName    = DMCAdditionalUIFieldsSvc:cInfoDomainIDForDateField()
    pbttField.IsValidDomain = lIsValidDomain( pbttField.DomainName )
    .
  
  return.
  
end.

if    poVariableInfo:Has( 'Table':U )
  and poVariableInfo:Has( 'Field':U )
  and DMCAdditionalUIFieldsSvc:cFieldNameForMasterDataTable( poVariableInfo:GetCharacter( 'Table':U ) ) = poVariableInfo:GetCharacter( 'Field':U ) then
do:
  
  assign
    pbttField.TableName    = poVariableInfo:GetCharacter( 'Table':U )
    pbttField.IsValidTable = lIsValidTable( pbttField.TableName )
    pbttField.FieldName    = DMCAdditionalUIFieldsSvc:cFieldNameForMasterDataTable( pbttField.TableName )
    pbttField.IsValidField = lIsValidField( pbttField.TableName, pbttField.FieldName )
    .
  
  return.
  
end.

assign
  pbttField.TableName    = {&K_MISSING_PLACEHOLDER}
  pbttField.IsValidTable = no
  pbttField.FieldName    = {&K_MISSING_PLACEHOLDER}
  pbttField.IsValidField = no
  .

end procedure. /* initializeMissingInfoFieldSourceProperties */


procedure initizalieMissingFieldSourceProperties :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define parameter buffer pbttField for temp-table ttField.

define variable oVariableInfo as JsonObject no-undo.

if    pbttField.FieldName  > '':U
   or pbttField.DomainName > '':U then
  
  return.

oVariableInfo = oVariableDefinitionInfo( pbttField.OriginalName ).

if pbttField.IsInfoField then
  
  run initializeMissingInfoFieldSourceProperties( oVariableInfo, buffer pbttField ).

else if oVariableInfo:Has( 'Domain':U ) then
  
  assign 
    pbttField.DomainName    = oVariableInfo:GetCharacter( 'Domain':U )
    pbttField.IsValidDomain = lIsValidDomain( pbttField.DomainName )
    .

else
  
  assign
    pbttField.DomainName    = {&K_MISSING_PLACEHOLDER}
    pbttField.IsValidDomain = no
    .

end procedure. /* initizalieMissingFieldSourceProperties */


procedure showError :
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define input parameter poError as Error no-undo.

define variable iErrorCounter as integer   no-undo.

do iErrorCounter = 1 to poError:NumMessages:
  
  message
    poError:GetMessage(iErrorCounter) skip
    view-as alert-box
    error
    buttons ok
    in window ghDummyWindow.
  
end.

end procedure. /* showError */


/* ************************  Function Implementations ***************** */

function clInsertText returns longchar 
  ( pclText         as longchar,
    pclTextToInsert as longchar,
    piInsertOffset  as integer ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable clTargetText as longchar no-undo.

return ( if piInsertOffset < 2 then
           pclTextToInsert + pclText
         else
              substring( pclText, 1, piInsertOffset - 1 )
           + pclTextToInsert
           + substring( pclText, piInsertOffset ) ).

end function. /* clInsertText */


function cPromptForDomainOrNull returns character 
  ( pcDomain as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable cDomain as character no-undo
  format 'x(40)':U
  view-as fill-in
  size 65 by 1.

define button btnOk
  label 'Ok':T2
  size 20 by 1.14
  auto-go.

define button btnCancel
  label 'Cancel':T
  size 20 by 1.14
  auto-end-key.

define frame fDomain
    'Please enter domain:':T view-as text size 45 by 1 at row 1.5 col 3
    cDomain at row 2.5 col 3 no-label
    btnOk at row 4.5 col 16
    btnCancel at row 4.5 col 36
  with 1 down keep-tab-order overlay
    view-as dialog-box
    side-labels no-underline three-d 
    size 71 by 6.44
    title gcTitlePrefix + 'Enter Domain':T
    default-button btnOk
    cancel-button btnCancel.

on window-close of frame fDomain
  apply 'end-error' to self.

cDomain = pcDomain.

PromptForDomain:
do on error undo, throw:
  
  update unless-hidden
    cDomain
    btnOk
    btnCancel
    with frame fDomain
      in window ghDummyWindow
      .
  
  if not {fnarg
           pa_lReposIsDomainExisting
           "cDomain"} then
  do:
    
    message
      substitute( 'Domain &1 does not exist':U, quoter( cDomain, '~'':U ) ) skip
      view-as alert-box
      error
      buttons ok
      in window ghDummyWindow.
    
    undo, retry PromptForDomain.
    
  end.
  
  return cDomain.
  
end.

end function. /* cPromptForDomainOrNull */


function cPromptForTargetNameOrNull returns character 
  ( pcTargetName as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable cTargetName as character no-undo
  format 'x(40)':U
  view-as fill-in
  size 65 by 1.

define button btnOk
  label 'Ok':T2
  size 20 by 1.14
  auto-go.

define button btnCancel
  label 'Cancel':T
  size 20 by 1.14
  auto-end-key.

define frame fTargetName
    'Please enter target name:':T view-as text size 45 by 1 at row 1.5 col 3
    cTargetName at row 2.5 col 3 no-label
    btnOk at row 4.5 col 16
    btnCancel at row 4.5 col 36
  with 1 down keep-tab-order overlay
    view-as dialog-box
    side-labels no-underline three-d 
    size 71 by 6.44
    title gcTitlePrefix + 'Enter Target Name':T
    default-button btnOk
    cancel-button btnCancel.

on window-close of frame fTargetName
  apply 'end-error' to self.

cTargetName = pcTargetName.

PromptForTargetName:
do on error undo, throw:
  
  update unless-hidden
    cTargetName
    btnOk
    btnCancel
    with frame fTargetName
      in window ghDummyWindow
      .
  
  if cTargetName = '':U then
  do:
    
    message
      'Target name may not be empty':U skip
      view-as alert-box
      error
      buttons ok
      in window ghDummyWindow.
    
    undo, retry PromptForTargetName.
    
  end.
  
  return cTargetName.
  
end.

end function. /* cPromptForTargetNameOrNull */


function cStripNamePrefix returns character 
  ( pcName as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable oMatch as Match no-undo.

oMatch = goCoreNameRegex:Match( pcName ).

if not oMatch:Success then
  return pcName.

return oMatch:Groups['CoreName':U]:value.

end function. /* cStripNamePrefix */


function cTargetName returns character 
  ( pcOriginalName as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable cCoreName as character no-undo.

cCoreName = cStripNamePrefix( pcOriginalName ).

return ( if cCoreName begins gcCustomLevel then
           cCoreName
         else
           lc( gcCustomLevel ) + cCoreName ).
  
end function. /* cTargetName */


function iGetLineNumberOfOffset returns integer 
  ( pclText  as longchar,
    piOffset as integer ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable clTextTillOffset as longchar        no-undo.
define variable oLineNoMatches   as MatchCollection no-undo.

clTextTillOffset = substring( pclText, 1, piOffset ).

oLineNoMatches = goLineNoRegex:Matches( clTextTillOffset ).

return oLineNoMatches:count + 1.

end function. /* iGetLineNumberOfOffset */


function iNextFieldOrderNumber returns integer 
  (  ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define buffer bttDRC_AddUIField for temp-table ttDRC_AddUIField.

find last bttDRC_AddUIField
  use-index ord
  no-error.

if not available bttDRC_AddUIField then
  return 10.

return ( if bttDRC_AddUIField.FieldOrderNumber mod 10 = 0 then
           bttDRC_AddUIField.FieldOrderNumber + 10
         else
           integer( truncate( bttDRC_AddUIField.FieldOrderNumber / 10, 0 ) * 10 + 10 ) ).

end function. /* iNextFieldOrderNumber */


function lCanImport returns logical 
  ( buffer pbttField for ttField ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

if pbttField.IsAddDBFieldExisting then
  
  return no.

if pbttField.DomainName > '':U then
  
  return lIsValidDomain(pbttField.DomainName).

if    pbttField.TableName > '':U
  and pbttField.FieldName > '':U then
  
  return lIsValidField( pbttField.TableName, pbttField.FieldName ).

return no.

end function. /* lCanImport */


function lIsValidDomain returns logical 
  ( pcDomain as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

if pcDomain = '':U then
  return no.

return {fnarg
         pa_lReposIsDomainExisting
         "pcDomain"}.

end function. /* lIsValidDomain */


function lIsValidField returns logical 
  ( pcTable as character,
    pcField as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

if    pcTable = '':U
   or pcField = '':U then
   
   return no.

if not lIsValidTable( pcTable ) then
  return no.

return (     {fnarg
               pa_lReposIsFieldExisting
               "pcTable,
                pcField"}
          or {fnarg
               pa_lReposIsAddDBFieldExisting
               "pcTable,
                pcField"} ).

end function. /* lIsValidField */

function lIsValidTable returns logical
  ( pcTable as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

if pcTable = '':U then
  return no.

return {fnarg
         pa_lReposIsTableExisting
         "pcTable"}.

end function. /* lIsValidTable */

function lPromptForFieldSelection returns logical 
  (  ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define buffer bttField for temp-table ttField.

define query brFields
  for bttField
  scrolling.

define browse brFields
  query brFields no-lock
    display
      bttField.IsSelected
        width 10
        column-label '':U
        view-as toggle-box
      bttField.LineNo
        width 10
      bttField.OriginalName
        width 25
      bttField.TargetName
        width 25
      bttField.TableName
      bttField.FieldName
      bttField.DomainName
        width 20
      bttField.IsReadOnly
        width 10
        view-as toggle-box
      with no-assign size 166 by 20 fit-last-column.

define button btnOk
  label 'Ok':T2
  size 20 by 1.14
  auto-go.

define button btnCancel
  label 'Cancel':T
  size 20 by 1.14
  auto-end-key.

define frame fSelectFields
  brFields at row 1.5 col 3
  btnOk at row 22.5 col 66
  btnCancel at row 22.5 col 86
  with 1 down keep-tab-order overlay
    view-as dialog-box 
    title gcTitlePrefix + 'Select Fields':U
    side-labels no-underline three-d 
    size 171 by 24.44
    default-button btnOk
    cancel-button btnCancel.

define menu m_PopupMenu
  menu-item mi_SetTargetName
    label "Set Target Name"
  rule
  menu-item mi_SetDomain
    label "Set Domain"
  menu-item mi_SetField
    label "Set Field"
    .

on window-close of frame fSelectFields
  apply 'end-error' to self.

on row-display of brFields in frame fSelectFields
do:
  
  if not bttField.CanImport then
    
    assign
      bttField.OriginalName:fgcolor in browse brFields = {&K_COLOR_DARK_GREY}
      bttField.TargetName:fgcolor   in browse brFields = {&K_COLOR_DARK_GREY}
      bttField.TableName:fgcolor    in browse brFields = {&K_COLOR_DARK_GREY}
      bttField.FieldName:fgcolor    in browse brFields = {&K_COLOR_DARK_GREY}
      bttField.DomainName:fgcolor   in browse brFields = {&K_COLOR_DARK_GREY}
      .
  
  if bttField.IsAddDBFieldExisting then
    
    bttField.TargetName:fgcolor in browse brFields = {&K_COLOR_DARK_GREEN}.
  
  if not bttField.IsValidTable then
    
    bttField.TableName:fgcolor in browse brFields = {&K_COLOR_RED}.
  
  if not bttField.IsValidField then
    
    bttField.FieldName:fgcolor in browse brFields = {&K_COLOR_RED}.
  
  if not bttField.IsValidDomain then
    
    bttField.DomainName:fgcolor in browse brFields = {&K_COLOR_RED}.
  
end.

on default-action of brFields in frame fSelectFields
do:
  
  if not available bttField then
    return.
  
  if bttField.IsAddDBFieldExisting then
    undo, throw new AppError( substitute( 'Field &1 already exists as additional db field':U,
                                          quoter( bttField.TargetName, '~'':U ) ),
                              -1 ).
  
  if    bttField.FieldName > '':U
    and not bttField.IsValidField then
    
    apply 'choose':U to menu-item mi_SetField in menu m_PopupMenu.
  
  if    bttField.DomainName > '':U
    and not bttField.IsValidDomain then
     
    apply 'choose':U to menu-item mi_SetDomain in menu m_PopupMenu.
  
  if bttField.CanImport then
  do:
    
    bttField.IsSelected = not bttField.IsSelected.
    display unless-hidden bttField.IsSelected with browse brFields.
    
  end.
  
  catch oError as Error :
    run showError( oError ).
  end catch.
  
end.

on menu-drop of menu m_PopupMenu
do:
  
  assign
    menu-item mi_SetDomain:sensitive in menu m_PopupMenu
      = (     available bttField
          and bttField.FieldName = '':U )
    menu-item mi_SetField:sensitive in menu m_PopupMenu
      = (     available bttField
          and bttField.DomainName = '':U )
    .
  
end.

on choose of menu-item mi_SetTargetName in menu m_PopupMenu
do:
  
  define variable cTargetName as character no-undo.
  
  cTargetName = cPromptForTargetNameOrNull( bttField.TargetName ).
  
  if cTargetName > '':U then
  do:
    
    bttField.TargetName = cTargetName.
    
    if not can-find( DRC_AddUIField
                       where DRC_AddUIField.DRC_Instance_Obj  = gcDRC_Instance_Obj
                         and DRC_AddUIField.DRC_AddUIField_ID = bttField.TargetName ) then
      
      bttField.IsAddDBFieldExisting = no.
    
    else
      
      run initializeFieldFromExistingRepositoryRecord( buffer bttField ).
    
    validate bttField.
    
    browse brFields:refresh().
    
  end.
  
end.

on choose of menu-item mi_SetDomain in menu m_PopupMenu
do:
  
  define variable cDomain as character no-undo.
  
  cDomain = cPromptForDomainOrNull( bttField.DomainName ).
  
  if cDomain > '':U then
  do:
    
    assign
      bttField.DomainName    = cDomain
      bttField.IsValidDomain = lIsValidDomain( bttField.DomainName )
      bttField.CanImport     = lCanImport( buffer bttField )
      .
    
    validate bttField.
    
    browse brFields:refresh().
    
  end.
  
end.

on choose of menu-item mi_SetField in menu m_PopupMenu
do:
  
  define variable oField as JsonObject no-undo.
  
  oField = oPromptForFieldOrNull( bttField.TableName, bttField.FieldName ).
  
  if valid-object( oField ) then
  do:
    
    assign
      bttField.TableName    = oField:GetCharacter( 'Table':U )
      bttField.IsValidTable = lIsValidTable( bttField.TableName ) 
      bttField.FieldName    = oField:GetCharacter( 'Field':U )
      bttField.IsValidField = lIsValidField( bttField.TableName, bttField.FieldName ) 
      bttField.CanImport     = lCanImport( buffer bttField )
      .
  
    validate bttField.
    
    browse brFields:refresh().
    
  end.
  
end.

brFields:popup-menu in frame fSelectFields = menu m_PopupMenu:handle.

open query brFields
  for each bttField
  .

update unless-hidden
  brFields
  btnOk
  btnCancel
  with frame fSelectFields
    top-only
    in window ghDummyWindow
    .

return yes.

end function. /* lPromptForFieldSelection */


function oExtractTableFieldInfoFromInfoVariableName returns JsonObject 
  ( pcFieldName as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable oMatch          as Match no-undo.
define variable oTableFieldInfo as JsonObject no-undo.

define variable cTable as character no-undo.
define variable cField as character no-undo.

assign
  oTableFieldInfo = new JsonObject()
  oMatch          = goTableInfoFieldRegex:match( pcFieldName )
  .

if not oMatch:Success then
  return oTableFieldInfo. 

assign
  cTable = oMatch:Groups['TableName':U]:Value
  cField = oMatch:Groups['FieldName':U]:Value
  .

oTableFieldInfo:Add( 'Table':U, cTable ). 
oTableFieldInfo:Add( 'Field':U, cField ). 

if lIsValidField( cTable, cField ) then
  
  oTableFieldInfo:Add( 'DataType':U,
                       {fnarg
                         pa_cReposFieldInformationByName
                         "cTable,
                          cField,
                          'Data-Type':U"} ).

return oTableFieldInfo.

end function. /* oExtractTableFieldInformationFromInfoVariableName */


function oPromptForFieldOrNull returns JsonObject 
  ( pcTable as character,
    pcField as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable oField as JsonObject no-undo.

define variable cTable as character no-undo
  format 'x(25)':U
  label 'Table':U
  view-as fill-in
  size 58 by 1.

define variable cField as character no-undo
  format 'x(32)':U
  label 'Field':U
  view-as fill-in
  size 58 by 1.

define button btnOk
  label 'Ok':T2
  size 20 by 1.14
  auto-go.

define button btnCancel
  label 'Cancel':T
  size 20 by 1.14
  auto-end-key.

define frame fField
    'Please enter field information:':T view-as text size 45 by 1 at row 1.5 col 8 colon-aligned
    cTable at row 2.5 col 8 colon-aligned
    cField at row 3.5 col 8 colon-aligned
    btnOk at row 5.5 col 16
    btnCancel at row 5.5 col 36
  with 1 down keep-tab-order overlay
    view-as dialog-box
    side-labels no-underline three-d 
    size 71 by 7.44
    title gcTitlePrefix + 'Enter Domain':T
    default-button btnOk
    cancel-button btnCancel.

on window-close of frame fField
  apply 'end-error' to self.

assign
  cTable = pcTable
  cField = pcField
  .

PromptForField:
do on error undo, throw:
  
  update unless-hidden
    cTable
    cField
    btnOk
    btnCancel
    with frame fField
      in window ghDummyWindow.
  
  if not lIsValidTable( cTable ) then
  do:
    
    message
      substitute( 'Table &1 does not exist':U, quoter( cTable, '~'':U ) ) skip
      view-as alert-box
      error
      buttons ok
      in window ghDummyWindow.
    
    undo, retry PromptForField.
    
  end.
  
  if not lIsValidField( cTable, cField ) then
  do:
    
    message
      substitute( 'Field &2 does not exist in Table &1':U,
                  quoter( cTable, '~'':U ),
                  quoter( cField, '~'':U ) ) skip
      view-as alert-box
      error
      buttons ok
      in window ghDummyWindow.
    
    undo, retry PromptForField.
    
  end.
  
  oField = new JsonObject().
  
  oField:add( 'Table':U, cTable ).
  oField:add( 'Field':U, cField ).
  
  return oField.
  
end.


end function. /* oPromptForFieldOrNull */


function oVariableDefinitionInfo returns JsonObject 
  ( pcOriginalName as character ):
/* Description ---------------------------------------------------------------*/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

define variable oVariableInfo as JsonObject no-undo.
define variable oMatch        as Match      no-undo.
define variable cReference    as character  no-undo.
define variable cTable        as character  no-undo.
define variable cField        as character  no-undo.

assign
  oVariableInfo = new JsonObject()
  oMatch        = Regex:Match( gclFileContent,
                               substitute( 'def(?:ine)?\s+var(?:iable)?\s+&1\s+(?:like\s+(?<ReferenceName>\S*)|as\s+(?<DataType>\S+))':U,
                                           pcOriginalName ),
                               RegexOptions:IgnoreCase or RegexOptions:Singleline )
  .

if not oMatch:Success then
  return oVariableInfo.

if oMatch:Groups['DataType':U]:Success then
  
  oVariableInfo:add( 'DataType':U,
                     oMatch:Groups['DataType':U]:Value ).

else if oMatch:Groups['ReferenceName':U]:Success then
do:
  
  cReference = oMatch:Groups['ReferenceName':U]:Value.
  
  if num-entries( cReference, '.':U ) >= 2 then
  do:
    
    cTable = entry( 1, cReference, '.':U ).
    oVariableInfo:add( 'Table':U, cTable ).
    
    cField = entry( 2, cReference, '.':U ).
    oVariableInfo:add( 'Field':U, cField ).
    
    if lIsValidField(cTable, cField ) then
      
      oVariableInfo:add( 'DataType':U,
                         {fnarg
                           pa_cReposFieldInformationByName
                           "cTable,
                            cField,
                            'Data-Type':U"} ).
      oVariableInfo:add( 'Domain':U,
                         {fnarg
                           pa_cReposFieldInformationByName
                           "cTable,
                            cField,
                            'DRC_Domain_ID':U"} ).
    
  end.
  
end.

return oVariableInfo.

end function. /* oVariableDefinitionInfo */
